<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Produk;
use App\Models\Transaksi;
use Illuminate\Support\Facades\DB;

class TransaksiController extends Controller
{
    public function barangMasuk()
    {
        $produks = Produk::all();
        return view('barang_masuk', compact('produks'));
    }

    public function storeBarangMasuk(Request $request)
    {
        $request->validate([
            'produk_id' => 'required|exists:produks,id',
            'jumlah' => 'required|integer|min:1',
            'tanggal' => 'required|date',
            'nama_bagian' => 'nullable|string|max:255',
            'keterangan' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            $transaksi = Transaksi::create([
                'kode_transaksi' => Transaksi::generateKodeTransaksi('masuk'),
                'jenis' => 'masuk',
                'produk_id' => $request->produk_id,
                'jumlah' => $request->jumlah,
                'tanggal' => $request->tanggal,
                'nama_bagian' => $request->nama_bagian,
                'keterangan' => $request->keterangan,
                'user_id' => auth()->id(),
            ]);

            // Update stok produk
            $produk = Produk::find($request->produk_id);
            $produk->updateStok($request->jumlah, 'masuk');

            DB::commit();

            return redirect()->back()->with('success', 'Barang masuk berhasil disimpan! Kode: ' . $transaksi->kode_transaksi);

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    public function barangKeluar()
    {
        $produks = Produk::where('stok', '>', 0)->get();
        return view('barang_keluar', compact('produks'));
    }

    public function storeBarangKeluar(Request $request)
    {
        $request->validate([
            'produk_id' => 'required|exists:produks,id',
            'jumlah' => 'required|integer|min:1',
            'tanggal' => 'required|date',
            'nama_bagian' => 'nullable|string|max:255',
            'keterangan' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            $produk = Produk::find($request->produk_id);
            
            // Cek stok mencukupi
            if (!$produk->stokCukup($request->jumlah)) {
                return redirect()->back()->with('error', 'Stok tidak mencukupi! Stok tersedia: ' . $produk->stok);
            }

            $transaksi = Transaksi::create([
                'kode_transaksi' => Transaksi::generateKodeTransaksi('keluar'),
                'jenis' => 'keluar',
                'produk_id' => $request->produk_id,
                'jumlah' => $request->jumlah,
                'tanggal' => $request->tanggal,
                'nama_bagian' => $request->nama_bagian,
                'keterangan' => $request->keterangan,
                'user_id' => auth()->id(),
            ]);

            // Update stok produk
            $produk->updateStok($request->jumlah, 'keluar');

            DB::commit();

            return redirect()->back()->with('success', 'Barang keluar berhasil disimpan! Kode: ' . $transaksi->kode_transaksi);

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    public function riwayatTransaksi()
    {
        $transaksis = Transaksi::with(['produk', 'user'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
        
        return view('riwayat_transaksi', compact('transaksis'));
    }
}
