<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\BiodataModel;

class Biodata extends BaseController
{
    protected $biodataModel;

    public function __construct()
    {
        $this->biodataModel = new BiodataModel();
    }

    /**
     * Display biodata list
     */
    public function index()
    {
        $data = [
            'title' => 'Data Biodata / CV',
            'biodata' => $this->biodataModel->findAll(),
        ];

        return view('admin/biodata/index', $data);
    }

    /**
     * Show create form
     */
    public function create()
    {
        $data = [
            'title' => 'Tambah Biodata',
        ];

        return view('admin/biodata/create', $data);
    }

    /**
     * Store new biodata
     */
    public function store()
    {
        $rules = [
            'nama_lengkap' => 'required|min_length[3]|max_length[100]',
            'alamat' => 'required',
            'email' => 'required|valid_email',
            'no_hp' => 'required|min_length[10]|max_length[20]',
            'foto' => 'max_size[foto,2048]|is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'nama_lengkap' => $this->request->getPost('nama_lengkap'),
            'jenis_kelamin' => $this->request->getPost('jenis_kelamin'),
            'tempat_lahir' => $this->request->getPost('tempat_lahir'),
            'tanggal_lahir' => $this->request->getPost('tanggal_lahir'),
            'agama' => $this->request->getPost('agama'),
            'alamat' => $this->request->getPost('alamat'),
            'email' => $this->request->getPost('email'),
            'no_hp' => $this->request->getPost('no_hp'),
            'deskripsi_diri' => $this->request->getPost('deskripsi_diri'),
        ];

        // Handle foto upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $newName = $foto->getRandomName();
            $foto->move(FCPATH . 'assets/uploads/biodata', $newName);
            $data['foto'] = $newName;
        }

        $this->biodataModel->insert($data);

        return redirect()->to('/admin/biodata')->with('success', 'Biodata berhasil ditambahkan!');
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $biodata = $this->biodataModel->find($id);

        if (!$biodata) {
            return redirect()->to('/admin/biodata')->with('error', 'Data tidak ditemukan.');
        }

        $data = [
            'title' => 'Edit Biodata',
            'biodata' => $biodata,
        ];

        return view('admin/biodata/edit', $data);
    }

    /**
     * Update biodata
     */
    public function update($id)
    {
        $biodata = $this->biodataModel->find($id);

        if (!$biodata) {
            return redirect()->to('/admin/biodata')->with('error', 'Data tidak ditemukan.');
        }

        $rules = [
            'nama_lengkap' => 'required|min_length[3]|max_length[100]',
            'alamat' => 'required',
            'email' => 'required|valid_email',
            'no_hp' => 'required|min_length[10]|max_length[20]',
            'foto' => 'max_size[foto,2048]|is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'nama_lengkap' => $this->request->getPost('nama_lengkap'),
            'jenis_kelamin' => $this->request->getPost('jenis_kelamin'),
            'tempat_lahir' => $this->request->getPost('tempat_lahir'),
            'tanggal_lahir' => $this->request->getPost('tanggal_lahir'),
            'agama' => $this->request->getPost('agama'),
            'alamat' => $this->request->getPost('alamat'),
            'email' => $this->request->getPost('email'),
            'no_hp' => $this->request->getPost('no_hp'),
            'deskripsi_diri' => $this->request->getPost('deskripsi_diri'),
        ];

        // Handle delete foto checkbox
        if ($this->request->getPost('hapus_foto')) {
            if ($biodata['foto'] && file_exists(FCPATH . 'assets/uploads/biodata/' . $biodata['foto'])) {
                unlink(FCPATH . 'assets/uploads/biodata/' . $biodata['foto']);
            }
            $data['foto'] = null;
        }

        // Handle foto upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            // Delete old foto
            if ($biodata['foto'] && file_exists(FCPATH . 'assets/uploads/biodata/' . $biodata['foto'])) {
                unlink(FCPATH . 'assets/uploads/biodata/' . $biodata['foto']);
            }
            $newName = $foto->getRandomName();
            $foto->move(FCPATH . 'assets/uploads/biodata', $newName);
            $data['foto'] = $newName;
        }

        $this->biodataModel->update($id, $data);

        return redirect()->to('/admin/biodata')->with('success', 'Biodata berhasil diperbarui!');
    }

    /**
     * Delete biodata
     */
    public function delete($id)
    {
        $biodata = $this->biodataModel->find($id);

        if (!$biodata) {
            return redirect()->to('/admin/biodata')->with('error', 'Data tidak ditemukan.');
        }

        // Delete associated files
        if ($biodata['foto'] && file_exists(FCPATH . 'assets/uploads/biodata/' . $biodata['foto'])) {
            unlink(FCPATH . 'assets/uploads/biodata/' . $biodata['foto']);
        }

        $this->biodataModel->delete($id);

        return redirect()->to('/admin/biodata')->with('success', 'Biodata berhasil dihapus!');
    }
    /**
     * Activate a biodata
     */
    public function activate($id)
    {
        // 1. Set all to inactive
        $this->biodataModel->where('is_active', 1)->set(['is_active' => 0])->update();

        // 2. Set selected to active
        $this->biodataModel->update($id, ['is_active' => 1]);

        return redirect()->to('/admin/biodata')->with('success', 'Biodata aktif berhasil diubah!');
    }
}
