<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ActivityModel;

class Activities extends BaseController
{
    protected $activityModel;

    public function __construct()
    {
        $this->activityModel = new ActivityModel();
    }

    public function index()
    {
        $keyword = $this->request->getGet('keyword');
        $sort    = $this->request->getGet('sort') ?? 'id';
        $order   = $this->request->getGet('order') ?? 'desc';

        $builder = $this->activityModel->orderBy($sort, $order);

        if ($keyword) {
            $builder->like('activity_name', $keyword)->orLike('date', $keyword);
        }

        $data = [
            'activities' => $builder->paginate(4),
            'pager'      => $this->activityModel->pager,
            'keyword'    => $keyword,
            'sort'       => $sort,
            'order'      => $order,
        ];

        return view('backend/activities/index', $data);
    }

    public function create()
    {
        return view('backend/activities/create');
    }

    public function store()
    {
        if (!$this->validate([
            'activity_name' => 'required',
            'date'          => 'required',
            'media'         => [
                'rules' => 'uploaded[media]|is_image[media]|mime_in[media,image/jpg,image/jpeg,image/png]|max_size[media,2048]',
                'errors' => [
                    'uploaded' => 'Silakan pilih gambar',
                    'is_image' => 'File yang diupload bukan gambar',
                    'mime_in'  => 'Hanya file jpg, jpeg, dan png yang diperbolehkan',
                    'max_size' => 'Ukuran maksimal 2MB'
                ]
            ]
        ])) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileMedia = $this->request->getFile('media');
        $fileName = $fileMedia->getRandomName();
        $fileMedia->move(FCPATH . 'uploads/activities', $fileName);

        $this->activityModel->save([
            'activity_name' => $this->request->getPost('activity_name'),
            'date'          => $this->request->getPost('date'),
            'media'         => $fileName
        ]);

        return redirect()->to('admin/activities')->with('success', 'Activity added successfully.');
    }

    public function edit($id)
    {
        $data['activity'] = $this->activityModel->find($id);
        if (!$data['activity']) {
            return redirect()->to('admin/activities')->with('error', 'Activity not found.');
        }
        return view('backend/activities/edit', $data);
    }

    public function update($id)
    {
        if (!$this->validate([
            'activity_name' => 'required',
            'date'          => 'required',
            'media'         => [
                'rules' => 'is_image[media]|mime_in[media,image/jpg,image/jpeg,image/png]|max_size[media,2048]',
                'errors' => [
                    'is_image' => 'File yang diupload bukan gambar',
                    'mime_in'  => 'Hanya file jpg, jpeg, dan png yang diperbolehkan',
                    'max_size' => 'Ukuran maksimal 2MB'
                ]
            ]
        ])) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'activity_name' => $this->request->getPost('activity_name'),
            'date'          => $this->request->getPost('date'),
        ];

        $fileMedia = $this->request->getFile('media');
        if ($fileMedia && $fileMedia->isValid() && !$fileMedia->hasMoved()) {
            $fileName = $fileMedia->getRandomName();
            $fileMedia->move(FCPATH . 'uploads/activities', $fileName);
            $data['media'] = $fileName;
        }

        $this->activityModel->update($id, $data);

        return redirect()->to('admin/activities')->with('success', 'Activity updated successfully.');
    }

    public function delete($id)
    {
        $this->activityModel->delete($id);
        return redirect()->to('admin/activities')->with('success', 'Activity deleted successfully.');
    }
}
