<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\GaleriModel;

class GaleriController extends BaseController
{
    protected $galeriModel;

    public function __construct()
    {
        $this->galeriModel = new GaleriModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search');
        $tipe = $this->request->getGet('tipe');
        $page = $this->request->getGet('page') ?? 1;

        $galeri = $this->galeriModel
            ->getGaleri($search, $tipe)
            ->paginate(12, 'default', $page);
        $pager = $this->galeriModel->pager;

        $data = [
            'title' => 'Kelola Galeri',
            'galeri' => $galeri,
            'pager' => $pager,
            'search' => $search,
            'tipe' => $tipe,
            'tipeList' => $this->galeriModel->getTipeList(),
        ];

        return view('admin/galeri/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Galeri',
            'tipeList' => $this->galeriModel->getTipeList(),
        ];

        return view('admin/galeri/create', $data);
    }

    public function store()
    {
        $rules = [
            'judul' => 'required|min_length[3]|max_length[255]',
            'tipe'  => 'required|in_list[foto,video]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tipe = $this->request->getPost('tipe');
        $data = [
            'judul'     => $this->request->getPost('judul'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'tipe'      => $tipe,
            'tanggal'   => $this->request->getPost('tanggal') ?: date('Y-m-d'),
        ];

        // Handle file upload
        $file = $this->request->getFile('file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            $newName = $file->getRandomName();
            $file->move(FCPATH . 'uploads/galeri', $newName);
            $data['file'] = $newName;

            // If video, also handle thumbnail
            if ($tipe === 'video') {
                $thumbnail = $this->request->getFile('thumbnail');
                if ($thumbnail && $thumbnail->isValid() && !$thumbnail->hasMoved()) {
                    $thumbName = 'thumb_' . $thumbnail->getRandomName();
                    $thumbnail->move(FCPATH . 'uploads/galeri', $thumbName);
                    $data['thumbnail'] = $thumbName;
                }
            }
        }

        $this->galeriModel->insert($data);

        return redirect()->to(base_url('admin/galeri'))->with('success', 'Media berhasil ditambahkan');
    }

    public function edit($id)
    {
        $galeri = $this->galeriModel->find($id);

        if (!$galeri) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Galeri tidak ditemukan');
        }

        $data = [
            'title' => 'Edit Galeri',
            'galeri' => $galeri,
            'tipeList' => $this->galeriModel->getTipeList(),
        ];

        return view('admin/galeri/edit', $data);
    }

    public function update($id)
    {
        $galeri = $this->galeriModel->find($id);

        if (!$galeri) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Galeri tidak ditemukan');
        }

        $rules = [
            'judul' => 'required|min_length[3]|max_length[255]',
            'tipe'  => 'required|in_list[foto,video]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tipe = $this->request->getPost('tipe');
        $data = [
            'judul'     => $this->request->getPost('judul'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'tipe'      => $tipe,
            'tanggal'   => $this->request->getPost('tanggal') ?: date('Y-m-d'),
        ];

        // Handle file upload
        $file = $this->request->getFile('file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            $newName = $file->getRandomName();
            $file->move(FCPATH . 'uploads/galeri', $newName);
            $data['file'] = $newName;

            // Delete old file
            if ($galeri['file'] && file_exists(FCPATH . 'uploads/galeri/' . $galeri['file'])) {
                unlink(FCPATH . 'uploads/galeri/' . $galeri['file']);
            }
        }

        // Handle thumbnail
        $thumbnail = $this->request->getFile('thumbnail');
        if ($thumbnail && $thumbnail->isValid() && !$thumbnail->hasMoved()) {
            $thumbName = 'thumb_' . $thumbnail->getRandomName();
            $thumbnail->move(FCPATH . 'uploads/galeri', $thumbName);
            $data['thumbnail'] = $thumbName;

            // Delete old thumbnail
            if ($galeri['thumbnail'] && file_exists(FCPATH . 'uploads/galeri/' . $galeri['thumbnail'])) {
                unlink(FCPATH . 'uploads/galeri/' . $galeri['thumbnail']);
            }
        }

        $this->galeriModel->update($id, $data);

        return redirect()->to(base_url('admin/galeri'))->with('success', 'Media berhasil diperbarui');
    }

    public function delete($id)
    {
        $galeri = $this->galeriModel->find($id);

        if (!$galeri) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Galeri tidak ditemukan');
        }

        // Delete files
        if ($galeri['file'] && file_exists(FCPATH . 'uploads/galeri/' . $galeri['file'])) {
            unlink(FCPATH . 'uploads/galeri/' . $galeri['file']);
        }
        if ($galeri['thumbnail'] && file_exists(FCPATH . 'uploads/galeri/' . $galeri['thumbnail'])) {
            unlink(FCPATH . 'uploads/galeri/' . $galeri['thumbnail']);
        }

        $this->galeriModel->delete($id);

        return redirect()->to(base_url('admin/galeri'))->with('success', 'Media berhasil dihapus');
    }
}
