<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\AktivitasModel;

class AktivitasController extends BaseController
{
    protected $aktivitasModel;

    public function __construct()
    {
        $this->aktivitasModel = new AktivitasModel();
    }

    public function index()
    {
        // Get query parameters
        $search = $this->request->getGet('search');
        $year = $this->request->getGet('year');
        $sort = $this->request->getGet('sort') ?? 'tanggal';
        $order = $this->request->getGet('order') ?? 'DESC';
        $page = $this->request->getGet('page') ?? 1;

        // Build query and paginate
        $aktivitas = $this->aktivitasModel
            ->getAktivitas($search, $year, $sort, $order)
            ->paginate(10, 'default', $page);
        $pager = $this->aktivitasModel->pager;

        $data = [
            'title' => 'Kelola Aktivitas',
            'aktivitas' => $aktivitas,
            'pager' => $pager,
            'search' => $search,
            'year' => $year,
            'sort' => $sort,
            'order' => $order,
            'availableYears' => $this->aktivitasModel->getAvailableYears(),
        ];

        return view('admin/aktivitas/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Aktivitas',
        ];

        return view('admin/aktivitas/create', $data);
    }

    public function store()
    {
        $rules = [
            'tanggal'        => 'required|valid_date',
            'jam'            => 'required',
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'tipe'           => 'required|in_list[foto,video]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tipe = $this->request->getPost('tipe');
        $data = [
            'tanggal'        => $this->request->getPost('tanggal'),
            'jam'            => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi'      => $this->request->getPost('deskripsi'),
            'tipe'           => $tipe,
        ];

        // Handle foto upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $newName = $foto->getRandomName();
            $foto->move(FCPATH . 'uploads/aktivitas', $newName);
            $data['foto'] = $newName;
        }

        // Handle video upload
        if ($tipe === 'video') {
            $video = $this->request->getFile('video');
            if ($video && $video->isValid() && !$video->hasMoved()) {
                $newName = $video->getRandomName();
                $video->move(FCPATH . 'uploads/aktivitas', $newName);
                $data['video'] = $newName;
            }

            // Handle thumbnail upload
            $thumbnail = $this->request->getFile('thumbnail');
            if ($thumbnail && $thumbnail->isValid() && !$thumbnail->hasMoved()) {
                $thumbName = 'thumb_' . $thumbnail->getRandomName();
                $thumbnail->move(FCPATH . 'uploads/aktivitas', $thumbName);
                $data['thumbnail'] = $thumbName;
            }
        }

        $this->aktivitasModel->insert($data);

        return redirect()->to(base_url('admin/aktivitas'))->with('success', 'Aktivitas berhasil ditambahkan');
    }

    public function edit($id)
    {
        $aktivitas = $this->aktivitasModel->find($id);

        if (!$aktivitas) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Aktivitas tidak ditemukan');
        }

        $data = [
            'title' => 'Edit Aktivitas',
            'aktivitas' => $aktivitas,
        ];

        return view('admin/aktivitas/edit', $data);
    }

    public function update($id)
    {
        $aktivitas = $this->aktivitasModel->find($id);

        if (!$aktivitas) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Aktivitas tidak ditemukan');
        }

        $rules = [
            'tanggal'        => 'required|valid_date',
            'jam'            => 'required',
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'tipe'           => 'required|in_list[foto,video]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tipe = $this->request->getPost('tipe');
        $data = [
            'tanggal'        => $this->request->getPost('tanggal'),
            'jam'            => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi'      => $this->request->getPost('deskripsi'),
            'tipe'           => $tipe,
        ];

        // Handle foto upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $newName = $foto->getRandomName();
            $foto->move(FCPATH . 'uploads/aktivitas', $newName);
            $data['foto'] = $newName;

            // Delete old photo if exists
            if ($aktivitas['foto'] && file_exists(FCPATH . 'uploads/aktivitas/' . $aktivitas['foto'])) {
                unlink(FCPATH . 'uploads/aktivitas/' . $aktivitas['foto']);
            }
        }

        // Handle video upload
        if ($tipe === 'video') {
            $video = $this->request->getFile('video');
            if ($video && $video->isValid() && !$video->hasMoved()) {
                $newName = $video->getRandomName();
                $video->move(FCPATH . 'uploads/aktivitas', $newName);
                $data['video'] = $newName;

                // Delete old video if exists
                if ($aktivitas['video'] && file_exists(FCPATH . 'uploads/aktivitas/' . $aktivitas['video'])) {
                    unlink(FCPATH . 'uploads/aktivitas/' . $aktivitas['video']);
                }
            }

            // Handle thumbnail upload
            $thumbnail = $this->request->getFile('thumbnail');
            if ($thumbnail && $thumbnail->isValid() && !$thumbnail->hasMoved()) {
                $thumbName = 'thumb_' . $thumbnail->getRandomName();
                $thumbnail->move(FCPATH . 'uploads/aktivitas', $thumbName);
                $data['thumbnail'] = $thumbName;

                // Delete old thumbnail if exists
                if ($aktivitas['thumbnail'] && file_exists(FCPATH . 'uploads/aktivitas/' . $aktivitas['thumbnail'])) {
                    unlink(FCPATH . 'uploads/aktivitas/' . $aktivitas['thumbnail']);
                }
            }
        }

        $this->aktivitasModel->update($id, $data);

        return redirect()->to(base_url('admin/aktivitas'))->with('success', 'Aktivitas berhasil diperbarui');
    }

    public function delete($id)
    {
        $aktivitas = $this->aktivitasModel->find($id);

        if (!$aktivitas) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Aktivitas tidak ditemukan');
        }

        // Delete photo if exists
        if ($aktivitas['foto'] && file_exists(FCPATH . 'uploads/aktivitas/' . $aktivitas['foto'])) {
            unlink(FCPATH . 'uploads/aktivitas/' . $aktivitas['foto']);
        }

        // Delete video if exists
        if ($aktivitas['video'] && file_exists(FCPATH . 'uploads/aktivitas/' . $aktivitas['video'])) {
            unlink(FCPATH . 'uploads/aktivitas/' . $aktivitas['video']);
        }

        // Delete thumbnail if exists
        if ($aktivitas['thumbnail'] && file_exists(FCPATH . 'uploads/aktivitas/' . $aktivitas['thumbnail'])) {
            unlink(FCPATH . 'uploads/aktivitas/' . $aktivitas['thumbnail']);
        }

        $this->aktivitasModel->delete($id);

        return redirect()->to(base_url('admin/aktivitas'))->with('success', 'Aktivitas berhasil dihapus');
    }
}

