<?php

if (!function_exists('media_url')) {
    /**
     * Get media URL with fallback
     */
    function media_url(?string $path, string $fallback = '/assets/img/placeholder.jpg'): string
    {
        if (empty($path)) {
            return base_url($fallback);
        }
        
        // Check if path is already a full URL
        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }
        
        // Build URL from uploads folder
        $fullPath = FCPATH . 'uploads/' . $path;
        if (file_exists($fullPath)) {
            return base_url('uploads/' . $path);
        }
        
        return base_url($fallback);
    }
}

if (!function_exists('upload_media')) {
    /**
     * Upload media file with validation
     */
    function upload_media($file, string $folder = 'images'): ?string
    {
        if (!$file->isValid() || $file->hasMoved()) {
            return null;
        }
        
        // Validate file type
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'application/pdf'];
        if (!in_array($file->getMimeType(), $allowedTypes)) {
            return null;
        }
        
        // Validate file size (max 5MB)
        if ($file->getSizeByUnit('mb') > 5) {
            return null;
        }
        
        // Create upload directory if not exists
        $uploadPath = FCPATH . 'uploads/' . $folder;
        if (!is_dir($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }
        
        // Generate unique filename
        $newName = $file->getRandomName();
        
        // Move file
        if ($file->move($uploadPath, $newName)) {
            return $folder . '/' . $newName;
        }
        
        return null;
    }
}

if (!function_exists('delete_media')) {
    /**
     * Delete media file
     */
    function delete_media(?string $path): bool
    {
        if (empty($path)) {
            return false;
        }
        
        $fullPath = FCPATH . 'uploads/' . $path;
        if (file_exists($fullPath)) {
            return unlink($fullPath);
        }
        
        return false;
    }
}

if (!function_exists('format_date')) {
    /**
     * Format date to Indonesian locale
     */
    function format_date(?string $date, string $format = 'd F Y'): string
    {
        if (empty($date)) {
            return '-';
        }
        
        $months = [
            'January'   => 'Januari',
            'February'  => 'Februari',
            'March'     => 'Maret',
            'April'     => 'April',
            'May'       => 'Mei',
            'June'      => 'Juni',
            'July'      => 'Juli',
            'August'    => 'Agustus',
            'September' => 'September',
            'October'   => 'Oktober',
            'November'  => 'November',
            'December'  => 'Desember',
        ];
        
        $formatted = date($format, strtotime($date));
        return strtr($formatted, $months);
    }
}

if (!function_exists('excerpt')) {
    /**
     * Create excerpt from HTML content
     */
    function excerpt(?string $content, int $length = 150): string
    {
        if (empty($content)) {
            return '';
        }
        
        // Strip HTML tags
        $text = strip_tags($content);
        
        // Trim whitespace
        $text = trim(preg_replace('/\s+/', ' ', $text));
        
        if (strlen($text) <= $length) {
            return $text;
        }
        
        return substr($text, 0, $length) . '...';
    }
}

if (!function_exists('get_setting')) {
    /**
     * Get setting value
     */
    function get_setting(string $key, $default = null)
    {
        static $settings = null;
        
        if ($settings === null) {
            $settingsModel = new \App\Models\SettingsModel();
            $settings = $settingsModel->getAllSettings();
        }
        
        return $settings[$key] ?? $default;
    }
}
