<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\EducationModel;

class EducationAdminController extends BaseController
{
    protected $educationModel;

    public function __construct()
    {
        $this->educationModel = new EducationModel();
        helper('media');
    }

    public function index()
    {
        $educations = $this->educationModel->orderBy('start_year', 'DESC')->findAll();

        $data = [
            'title'      => 'Kelola Pendidikan',
            'educations' => $educations,
        ];

        return view('admin/education/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Pendidikan',
        ];

        return view('admin/education/create', $data);
    }

    public function store()
    {
        $rules = [
            'institution' => 'required|max_length[255]',
            'start_year'  => 'permit_empty|exact_length[4]|numeric',
            'end_year'    => 'permit_empty|exact_length[4]|numeric',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'institution' => $this->request->getPost('institution'),
            'degree'      => $this->request->getPost('degree'),
            'major'       => $this->request->getPost('major'),
            'start_year'  => $this->request->getPost('start_year'),
            'end_year'    => $this->request->getPost('end_year'),
            'description' => $this->request->getPost('description'),
            'location'    => $this->request->getPost('location'),
        ];

        // Handle logo
        $logo = $this->request->getFile('logo');
        if ($logo && $logo->isValid() && !$logo->hasMoved()) {
            $data['logo'] = upload_media($logo, 'educations');
        }

        if ($this->educationModel->insert($data)) {
            return redirect()->to('/admin/educations')->with('success', 'Pendidikan berhasil ditambahkan!');
        }

        return redirect()->back()->withInput()->with('error', 'Gagal menambahkan pendidikan.');
    }

    public function edit($id)
    {
        $education = $this->educationModel->find($id);

        if (!$education) {
            return redirect()->to('/admin/educations')->with('error', 'Pendidikan tidak ditemukan.');
        }

        $data = [
            'title'     => 'Edit Pendidikan',
            'education' => $education,
        ];

        return view('admin/education/edit', $data);
    }

    public function update($id)
    {
        $education = $this->educationModel->find($id);

        if (!$education) {
            return redirect()->to('/admin/educations')->with('error', 'Pendidikan tidak ditemukan.');
        }

        $rules = [
            'institution' => 'required|max_length[255]',
            'start_year'  => 'permit_empty|exact_length[4]|numeric',
            'end_year'    => 'permit_empty|exact_length[4]|numeric',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'institution' => $this->request->getPost('institution'),
            'degree'      => $this->request->getPost('degree'),
            'major'       => $this->request->getPost('major'),
            'start_year'  => $this->request->getPost('start_year'),
            'end_year'    => $this->request->getPost('end_year'),
            'description' => $this->request->getPost('description'),
            'location'    => $this->request->getPost('location'),
        ];

        // Handle logo
        $logo = $this->request->getFile('logo');
        if ($logo && $logo->isValid() && !$logo->hasMoved()) {
            // Delete old logo
            if (!empty($education['logo'])) {
                delete_media($education['logo']);
            }
            $data['logo'] = upload_media($logo, 'educations');
        }

        if ($this->educationModel->update($id, $data)) {
            return redirect()->to('/admin/educations')->with('success', 'Pendidikan berhasil diperbarui!');
        }

        return redirect()->back()->withInput()->with('error', 'Gagal memperbarui pendidikan.');
    }

    public function delete($id)
    {
        $education = $this->educationModel->find($id);

        if (!$education) {
            return redirect()->to('/admin/educations')->with('error', 'Pendidikan tidak ditemukan.');
        }

        // Delete logo
        if (!empty($education['logo'])) {
            delete_media($education['logo']);
        }

        if ($this->educationModel->delete($id)) {
            return redirect()->to('/admin/educations')->with('success', 'Pendidikan berhasil dihapus!');
        }

        return redirect()->to('/admin/educations')->with('error', 'Gagal menghapus pendidikan.');
    }
}
