# Penjelasan Singkat Arsitektur & Alur Aplikasi D'Shop (Laravel 10/12.x)

## Framework & Fondasi
- **Laravel** (PHP 8.2): pola **MVC** dengan routing berbasis `routes/web.php` dan `routes/auth.php`.
- **Blade**: templating server-side untuk halaman publik, checkout, admin, kasir.
- **Authentication & Authorization**: multi-guard (`web`, `admin`, `cashier`) dengan middleware `auth:*`, `role:*`, dan pengecekan manual di controller (contoh `OrderController::authorizeOrder`). Redirect login diatur di `Authenticate` middleware.
- **Eloquent ORM**: model utama seperti `User`, `Order`, `OrderItem`, `Product`, `Category`, `Review`, `CartItem`, `Payment`, `UserAddress`, `StoreSetting`.
- **Jobs/Services**: layanan eksternal dibungkus service class, mis. `App\Services\Shipping\RajaOngkirService`, `ShipperService`, `PaymentGatewayService` (Midtrans).
- **Build Tooling & Frontend Stack**:
  - **Vite** (lihat `package.json` script `dev`/`build`) untuk bundling aset.
  - **Tailwind CSS** + plugin **@tailwindcss/forms** sebagai utilitas styling utama (kelas utilitas di Blade).
  - **Alpine.js** untuk interaksi ringan di frontend (toggle, dropdown, form quantity, sticky UI).
  - **Axios** tersedia; sebagian AJAX pakai `fetch` (contoh di checkout untuk pencarian kota & ongkir).
  - **PostCSS/Autoprefixer** untuk kompatibilitas CSS.
  - Aset di-load via `@vite(['resources/css/app.css','resources/js/app.js'])` (lihat layout).

## Routing Utama (web)
- **Halaman publik**: `/` (Landing), `/catalog`, `/products/{product:slug}`.
- **Cart & Checkout**: `/cart`, `/cart/buy-now`, `/checkout` (GET/POST).
- **Shipping API publik**: prefix `/api/shipping` untuk pencarian kota/provinsi & kalkulasi ongkir (JSON).
- **Customer (guard web)**: dashboard `/dashboard`, daftar pesanan `/orders`, profile, wishlist, alamat.
- **Order detail (multi-guard)**: `/orders/{order}`, `/orders/{order}/print`, `/orders/{order}/payment` bisa diakses `web|admin|cashier` dengan otorisasi tambahan di `OrderController` (admin/kasir bebas, customer hanya miliknya).
- **Admin (guard admin + role admin)**: prefix `/admin`, resource products/categories, orders, income, settings.
- **Kasir (guard cashier/admin + role)**: prefix `/kasir`, buat & kelola pesanan kasir.
- **Auth routes**: `/login` (user), `/admin/login`, `/kasir/login`, register, reset password, Google OAuth.

## Autentikasi Multi-Guard
- `config/auth.php`: guard `web`, `admin`, `cashier` memakai provider `users` (kolom `role` membedakan).
- **AdminLoginController** & **CashierLoginController** menggunakan guard spesifik (`authenticate('admin'|'cashier')`) dan memaksa role sesuai, logout jika salah.
- **AuthenticatedSessionController** (login user) menolak admin/kasir login di form user dan mengarahkan ke halaman login khusus.
- **Middleware Authenticate** mengarahkan ke login sesuai prefix URL (`/admin` → `admin.login`, `/kasir` → `cashier.login`, lainnya → `login`).

## Otorisasi Pesanan
- `OrderController::authorizeOrder`: 
  - Admin/kasir (guard admin/cashier) bebas akses semua order.
  - Customer (guard web) hanya order miliknya; jika bukan pemilik → 403.

## View/Blade
- **Layout**: `resources/views/layouts/admin.blade.php`, `components/admin/*`, `app.blade.php` (layout umum).
- **Admin**: folder `resources/views/admin/*` (products index/create/edit, categories, orders, dsb).
- **Frontend**: `resources/views/products/show.blade.php`, `products/index.blade.php`, `checkout/index.blade.php`, `cart/index.blade.php`, `orders/show.blade.php`, dll.
- **Auth views**: `resources/views/auth/login.blade.php`, `auth/admin-login.blade.php`, `auth/cashier-login.blade.php`, register/reset/email verify.
- **Interaksi JS (contoh)**:
  - Checkout: cari kota `/api/shipping/search-city`, hitung ongkir `/api/shipping/calculate-cost`, update ringkasan ongkir di DOM.
  - Produk: buy-now/add-to-cart, gallery preview, wishlist, toggle quantity via Alpine/fetch.
  - Admin: toast flash di layout, toggle sidebar/theme, form file upload.

## Model & Relasi Kunci
- **Product**: belongsTo `Category`, hasMany `ProductImage`, `OrderItem`, `Review`, accessor harga akhir/thumbnail.
- **Order**: hasMany `OrderItem`, hasMany `Payment`, belongsTo `User` (user_id), `cashier` (opsional).
- **Review**: belongsTo `Product`, `User`; scope `approved`, `recent`.
- **Cart**: (CartService) menyimpan `CartItem` per user/session.

## Service Eksternal
- **RajaOngkirService**: cari provinsi/kota, ongkir; fallback offline dataset (`resources/data/rajaongkir_*`). Konfigurasi `.env`: `RAJAONGKIR_API_KEY`, `RAJAONGKIR_ACCOUNT_TYPE`, `RAJAONGKIR_BASE_URL`, `RAJAONGKIR_SKIP_SSL_VERIFY`, `RAJAONGKIR_OFFLINE_COST`.
- **ShipperService** (jika diisi): alternatif ongkir via Shipper.
- **PaymentGatewayService (Midtrans)**: membuat Snap token, redirect ke pembayaran. `.env`: `MIDTRANS_SERVER_KEY`, `MIDTRANS_CLIENT_KEY`, `MIDTRANS_IS_PRODUCTION`.

## Controller Penting
- **CheckoutController**: tampil checkout, buat Order + Payment, simpan alamat default, redirect ke payment jika Midtrans.
- **CartController**: tambah/update/hapus item, buy-now.
- **ProductController**: listing & detail produk + ulasan (ReviewController store/destroy).
- **AdminProductController**: CRUD produk, upload thumbnail/galeri.
- **ShippingController**: endpoint JSON untuk search kota, get provinsi/kota, hitung ongkir (RajaOngkir/Shipper).

## Middleware & Role
- **RoleMiddleware**: memeriksa `role` terhadap parameter (admin/cashier/user). Digunakan di route admin/kasir.
- **VerifyEmail**: semua route dalam grup `auth, verified` memerlukan email terverifikasi.

## Tools/Helper
- **Cache**: digunakan di RajaOngkirService untuk cache city/province.
- **Storage**: symlink `public/storage` → `storage/app/public` diperlukan untuk gambar produk.
- **Validation**: form request `LoginRequest` untuk login; inline `$request->validate` untuk banyak form.
- **Env/Config Penting**:
  - `APP_URL` sesuai domain (mis. https://dshop.web.id).
  - Shipping: `RAJAONGKIR_API_KEY`, `RAJAONGKIR_ACCOUNT_TYPE`, `RAJAONGKIR_BASE_URL`, `RAJAONGKIR_SKIP_SSL_VERIFY`, `RAJAONGKIR_OFFLINE_COST`.
  - Payment: `MIDTRANS_SERVER_KEY`, `MIDTRANS_CLIENT_KEY`, `MIDTRANS_IS_PRODUCTION`.
  - Auth: guard di `config/auth.php` (`web`, `admin`, `cashier`), redirect middleware `Authenticate`.
  - Aset: jalankan `npm install` lalu `npm run build` (atau `npm run dev`) agar CSS/JS siap pakai.

## Alur Login Per Role
1. **User**: /login → guard `web` → jika role admin/kasir, diarahkan ke login khusus.
2. **Admin**: /admin/login → guard `admin` → cek role admin → dashboard admin.
3. **Kasir**: /kasir/login → guard `cashier` → cek role cashier/admin → dashboard kasir.

## Tips Deploy/Operasional
- Setelah mengubah route/auth, jalankan `php artisan optimize:clear` (config/routing cache).
- Pastikan `.env` sesuai domain: `APP_URL=https://dshop.web.id`.
- Buat symlink storage di server: `ln -s storage/app/public public/storage` (jika `storage:link` diblokir).
- Cek role user di DB (kolom `role`) untuk akses admin/kasir.
- Untuk akses order tanpa 403: login sesuai role atau sebagai pemilik order.

## File yang Sering Diedit
- `routes/web.php`, `routes/auth.php`
- `app/Http/Controllers/Auth/*LoginController.php`, `OrderController.php`, `CheckoutController.php`, `ShippingController.php`
- `resources/views/checkout/index.blade.php`, `products/show.blade.php`, `admin/products/*.blade.php`, `auth/*.blade.php`
- `app/Services/Shipping/RajaOngkirService.php`
- `config/auth.php`, `config/services.php`
