<?php

namespace App\Services\Shipping;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ShipperService
{
    protected string $baseUrl;
    protected string $accessToken;
    protected string $clientId;
    protected string $clientSecret;
    protected string $pricingPath;
    protected string $areasPath;

    public function __construct()
    {
        $cfg = config('services.shipper');
        $this->baseUrl = rtrim((string)($cfg['base_url'] ?? ''), '/');
        $this->accessToken = (string)($cfg['access_token'] ?? '');
        $this->clientId = (string)($cfg['client_id'] ?? '');
        $this->clientSecret = (string)($cfg['client_secret'] ?? '');
        $this->pricingPath = (string)($cfg['pricing_path'] ?? '/v3/pricing/domestic');
        $this->areasPath = (string)($cfg['areas_path'] ?? '/v3/locations/areas');
    }

    public function isConfigured(): bool
    {
        return !empty($this->baseUrl) && (!empty($this->accessToken) || (!empty($this->clientId) && !empty($this->clientSecret)));
    }

    protected function authHeaders(): array
    {
        $headers = [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ];
        if (!empty($this->accessToken)) {
            $headers['Authorization'] = 'Bearer ' . $this->accessToken;
        }
        return $headers;
    }

    public function searchArea(string $query): array
    {
        if (!$this->isConfigured()) {
            return [];
        }
        try {
            $url = $this->baseUrl . $this->areasPath;
            $resp = Http::withHeaders($this->authHeaders())->get($url, ['query' => $query, 'limit' => 10]);
            if ($resp->successful()) {
                $json = $resp->json();
                // Normalize: expect an array under 'data' or root
                $items = $json['data'] ?? $json ?? [];
                return is_array($items) ? $items : [];
            }
            Log::warning('Shipper searchArea failed', ['status' => $resp->status(), 'body' => $resp->body()]);
        } catch (\Throwable $e) {
            Log::error('Shipper searchArea exception: ' . $e->getMessage());
        }
        return [];
    }

    public function findAreaIdByQuery(string $query): ?int
    {
        $areas = $this->searchArea($query);
        if (!$areas) return null;
        $first = $areas[0];
        $id = $first['area_id'] ?? $first['id'] ?? null;
        return $id ? (int)$id : null;
    }

    public function getRatesByAreaId(int $originAreaId, int $destinationAreaId, int $weight, array $couriers = []): array
    {
        if (!$this->isConfigured()) {
            return ['success' => false, 'data' => [], 'message' => 'Shipper not configured'];
        }
        try {
            $url = $this->baseUrl . $this->pricingPath;
            $payload = [
                'origin' => ['area_id' => $originAreaId],
                'destination' => ['area_id' => $destinationAreaId],
                'couriers' => $couriers ?: null,
                'weight' => max($weight, 1),
            ];
            $resp = Http::withHeaders($this->authHeaders())->post($url, array_filter($payload, fn($v) => $v !== null));
            if ($resp->successful()) {
                $json = $resp->json();
                $rows = $json['data'] ?? $json['rates'] ?? [];
                $out = [];
                foreach ((array)$rows as $r) {
                    // Attempt to normalize common fields
                    $out[] = [
                        'courier_code' => $r['courier_code'] ?? $r['courier'] ?? ($r['logistic']['code'] ?? ''),
                        'courier_name' => $r['courier_name'] ?? ($r['logistic']['name'] ?? ''),
                        'service' => $r['service'] ?? ($r['service_name'] ?? 'REG'),
                        'description' => $r['notes'] ?? ($r['description'] ?? ''),
                        'cost' => (int)($r['price'] ?? $r['total_price'] ?? $r['cost'] ?? 0),
                        'etd' => (string)($r['etd'] ?? $r['estimation'] ?? '-'),
                    ];
                }
                return ['success' => !empty($out), 'data' => $out];
            }
            Log::warning('Shipper getRates non-200', ['status' => $resp->status(), 'body' => $resp->body()]);
        } catch (\Throwable $e) {
            Log::error('Shipper getRates exception: ' . $e->getMessage());
        }
        return ['success' => false, 'data' => [], 'message' => 'Failed to get shipper rates'];
    }
}
