<?php

namespace App\Services\Shipping;

use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class RajaOngkirService
{
    protected string $apiKey;
    protected string $baseUrl;
    protected string $accountType; // starter, basic, pro
    protected ?string $cityLookupModeCache = null;
    protected bool $offlineCostEnabled = false;
    protected bool $skipSslVerify = false;

    public function __construct()
    {
        $this->apiKey = (string) config('services.rajaongkir.api_key', '');
        $this->accountType = (string) config('services.rajaongkir.account_type', 'starter');

        $configuredBaseUrl = (string) config('services.rajaongkir.base_url');

        $this->baseUrl = $configuredBaseUrl ?: match ($this->accountType) {
            'pro' => 'https://pro.rajaongkir.com/api',
            'basic' => 'https://api.rajaongkir.com/basic',
            default => 'https://api.rajaongkir.com/starter',
        };

        // Feature toggles
        $this->offlineCostEnabled = (bool) (config('services.rajaongkir.offline_cost', env('RAJAONGKIR_OFFLINE_COST', false)));
        $this->skipSslVerify = (bool) (config('services.rajaongkir.skip_ssl_verify', env('RAJAONGKIR_SKIP_SSL_VERIFY', false)));
    }

    public function getProvinces(): array
    {
        $cacheKey = 'ro_provinces_' . md5(($this->baseUrl ?: 'auto') . '|' . ($this->accountType ?: 'starter'));

        return Cache::remember($cacheKey, now()->addDays(7), function () {
            $provinces = $this->fetchProvincesFromApi();
            if (!empty($provinces)) {
                return $provinces;
            }
            return $this->loadOfflineProvinces();
        });
    }

    public function getCities($provinceId = null): array
    {
        $suffix = $provinceId ? "province_{$provinceId}" : 'all';
        $cacheKey = 'ro_cities_' . $suffix . '_' . md5(($this->baseUrl ?: 'auto') . '|' . ($this->accountType ?: 'starter'));

        return Cache::remember($cacheKey, now()->addDays(7), function () use ($provinceId) {
            $cities = $this->fetchCitiesFromApi($provinceId);
            if (empty($cities)) {
                $cities = $this->loadOfflineCities();
            }
            if ($provinceId && !empty($cities)) {
                return array_values(array_filter($cities, function ($city) use ($provinceId) {
                    return (int) $city['province_id'] === (int) $provinceId;
                }));
            }
            return $cities;
        });
    }

    public function getCost($origin, $destination, $weight, $courier = 'jne'): array
    {
        try {
            $payload = [
                'origin' => $origin,
                'destination' => $destination,
                'weight' => $weight,
                'courier' => $courier,
            ];

            // Primary: POST as form per docs
            $response = $this->client([
                'key' => $this->apiKey,
                'Accept' => 'application/json',
            ])->asForm()->post("{$this->baseUrl}/cost", $payload);

            // Fallback: GET when some gateways respond 405
            if (!$response->successful() && $response->status() === 405) {
                $response = $this->client(['key' => $this->apiKey, 'Accept' => 'application/json'])
                    ->get("{$this->baseUrl}/cost", $payload);
            }

            // Additional fallback for Komerce keys / environments
            if (!$response->successful()) {
                $lowerBase = strtolower($this->baseUrl);
                $isKomerce = str_contains($lowerBase, 'komerce.id');

                if ($isKomerce) {
                    // Try multiple endpoint shapes commonly used by Komerce
                    $candidates = [
                        rtrim($this->baseUrl, '/') . '/cost',
                        rtrim($this->baseUrl, '/') . '/cost/domestic',
                        rtrim($this->baseUrl, '/') . '/domestic/cost',
                        rtrim($this->baseUrl, '/') . '/domestic-cost',
                        rtrim($this->baseUrl, '/') . '/costs/domestic',
                        'https://collaborator.komerce.id/rajaongkir/cost',
                    ];

                    foreach ($candidates as $endpoint) {
                        // First try POST form
                        $response = $this->client(['key' => $this->apiKey, 'Accept' => 'application/json'])
                            ->asForm()->post($endpoint, $payload);
                        if ($response->successful() && $this->responseLooksJson($response)) {
                            break;
                        }
                        // If not ok, try GET
                        $response = $this->client([
                            'key' => $this->apiKey,
                            'Accept' => 'application/json',
                        ])->get($endpoint, $payload);
                        if ($response->successful() && $this->responseLooksJson($response)) {
                            break;
                        }

                        // Try Authorization: Bearer style
                        $response = $this->client([
                            'Authorization' => 'Bearer ' . $this->apiKey,
                            'Accept' => 'application/json',
                        ])->asForm()->post($endpoint, $payload);
                        if ($response->successful() && $this->responseLooksJson($response)) {
                            break;
                        }
                        $response = $this->client([
                            'Authorization' => 'Bearer ' . $this->apiKey,
                            'Accept' => 'application/json',
                        ])->get($endpoint, $payload);
                        if ($response->successful() && $this->responseLooksJson($response)) {
                            break;
                        }
                    }
                } else {
                    // If base is not Komerce but server says 410/404/405, try collaborator as last resort
                    if (in_array($response->status(), [410, 404, 405])) {
                        // Try known collaborator API hosts and endpoint shapes
                        $fallbackHosts = [
                            'https://api.collaborator.komerce.id',
                            'https://api.collaborator.komerce.my.id',
                            'https://collaborator.komerce.id',
                        ];
                        $paths = [
                            '/rajaongkir/cost',
                            '/v1/rajaongkir/cost',
                            '/raja-ongkir/cost',
                            '/api/v1/rajaongkir/cost',
                            '/rajaongkir/costs/domestic',
                            '/domestic/cost',
                        ];
                        foreach ($fallbackHosts as $host) {
                            foreach ($paths as $path) {
                                $endpoint = rtrim($host, '/') . $path;
                                $response = $this->client(['key' => $this->apiKey, 'Accept' => 'application/json'])
                                    ->asForm()->post($endpoint, $payload);
                                if ($response->successful() && $this->responseLooksJson($response)) { break 2; }
                                $response = $this->client(['key' => $this->apiKey, 'Accept' => 'application/json'])
                                    ->get($endpoint, $payload);
                                if ($response->successful() && $this->responseLooksJson($response)) { break 2; }
                                $response = $this->client(['Authorization' => 'Bearer ' . $this->apiKey, 'Accept' => 'application/json'])
                                    ->asForm()->post($endpoint, $payload);
                                if ($response->successful() && $this->responseLooksJson($response)) { break 2; }
                                $response = $this->client(['Authorization' => 'Bearer ' . $this->apiKey, 'Accept' => 'application/json'])
                                    ->get($endpoint, $payload);
                                if ($response->successful() && $this->responseLooksJson($response)) { break 2; }
                            }
                        }
                    }
                }
            }

            if ($response->successful() && $this->responseLooksJson($response)) {
                $data = $response->json();
                $results = $data['rajaongkir']['results'] ?? [];

                $costs = [];
                // RajaOngkir format
                foreach ($results as $result) {
                    if (!empty($result['costs'])) {
                        foreach ($result['costs'] as $cost) {
                            $costs[] = [
                                'courier_code' => $result['code'],
                                'courier_name' => $result['name'],
                                'service' => $cost['service'],
                                'description' => $cost['description'] ?? '',
                                'cost' => $cost['cost'][0]['value'] ?? 0,
                                'etd' => $cost['cost'][0]['etd'] ?? '-',
                                'note' => $cost['cost'][0]['note'] ?? '',
                            ];
                        }
                    }
                }

                // Komerce possible format
                if (empty($costs) && isset($data['data']) && is_array($data['data'])) {
                    foreach ($data['data'] as $row) {
                        $costs[] = [
                            'courier_code' => $row['courier_code'] ?? ($row['courier'] ?? ($row['logistic']['code'] ?? '')),
                            'courier_name' => $row['courier_name'] ?? ($row['logistic']['name'] ?? ''),
                            'service' => $row['service'] ?? ($row['service_name'] ?? 'REG'),
                            'description' => $row['description'] ?? ($row['notes'] ?? ''),
                            'cost' => (int)($row['cost'] ?? $row['price'] ?? $row['total_price'] ?? 0),
                            'etd' => (string)($row['etd'] ?? $row['estimation'] ?? '-'),
                            'note' => $row['note'] ?? '',
                        ];
                    }
                }

                return [
                    'success' => !empty($costs),
                    'data' => $costs,
                ];
            }

            // As a last-resort fallback for local/dev, try offline cost dataset
            $offlineCosts = $this->offlineCostEnabled ? $this->loadOfflineCosts($origin, $destination, $weight, $courier) : [];
            if (!empty($offlineCosts) && $this->offlineCostEnabled) {
                return [
                    'success' => true,
                    'data' => $offlineCosts,
                ];
            }

            Log::error('RajaOngkir Get Cost Error', [
                'origin' => $origin,
                'destination' => $destination,
                'weight' => $weight,
                'courier' => $courier,
                'status' => method_exists($response, 'status') ? $response->status() : null,
                'body' => method_exists($response, 'body') ? $response->body() : null,
            ]);

            return [
                'success' => false,
                'message' => 'Gagal mendapatkan biaya pengiriman',
                'data' => [],
            ];
        } catch (\Exception $e) {
            Log::error('RajaOngkir Get Cost Exception: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => $e->getMessage(),
                'data' => [],
            ];
        }
    }

    public function getMultipleCourierCosts($origin, $destination, $weight, array $couriers = ['jne', 'pos', 'tiki']): array
    {
        $all = [];
        foreach ($couriers as $c) {
            $r = $this->getCost($origin, $destination, $weight, $c);
            if (!empty($r['success'])) {
                $all = array_merge($all, $r['data']);
            }
        }
        return ['success' => !empty($all), 'data' => $all];
    }

    public function searchCity($cityName): array
    {
        $cities = $this->getCities();
        if (empty($cities)) {
            Log::warning('RajaOngkir: No cities available for search.');
            return [];
        }
        $filtered = collect($cities)->filter(function ($city) use ($cityName) {
            $searchIn = strtolower($city['city_name']);
            $term = strtolower($cityName);
            $full = strtolower(($city['type'] ?? '') . ' ' . $city['city_name']);
            return str_contains($searchIn, $term) || str_contains($full, $term);
        })->values();

        return $filtered->sortBy(function ($city) use ($cityName) {
            $searchIn = strtolower($city['city_name']);
            $full = strtolower(($city['type'] ?? '') . ' ' . $city['city_name']);
            $term = strtolower($cityName);
            if ($searchIn === $term || $full === $term) return 0;
            if (str_starts_with($searchIn, $term) || str_starts_with($full, $term)) return 1;
            return 2;
        })->values()->all();
    }

    public function getCityById($cityId)
    {
        $cities = $this->getCities();
        foreach ($cities as $c) {
            if ((int) ($c['city_id'] ?? 0) === (int) $cityId) return $c;
        }
        return null;
    }

    public function isConfigured(): bool
    {
        return !empty($this->apiKey);
    }

    public function cityLookupMode(): string
    {
        // Simplify: treat as 'live' when configured; never display 'offline' mode in UI
        if ($this->cityLookupModeCache !== null) return $this->cityLookupModeCache;
        if ($this->isConfigured()) return $this->cityLookupModeCache = 'live';
        return $this->cityLookupModeCache = 'disabled';
    }

    public function canSearchCities(): bool
    {
        return $this->cityLookupMode() !== 'disabled';
    }

    public function isShippingEstimatorAvailable(): bool
    {
        return $this->cityLookupMode() === 'live';
    }

    public function hasOfflineCityDataset(): bool
    {
        return File::exists(resource_path('data/rajaongkir_cities.json'));
    }

    protected function fetchProvincesFromApi(): array
    {
        if (!$this->isConfigured()) return [];
        try {
            $response = $this->client(['key' => $this->apiKey])->timeout(5)->get("{$this->baseUrl}/province");
        } catch (\Throwable $e) {
            Log::warning('RajaOngkir provinces unreachable', ['message' => $e->getMessage()]);
            return [];
        }
        $results = $this->extractResults($response);
        if (empty($results)) {
            Log::warning('RajaOngkir provinces returned empty/invalid payload', [
                'status' => $response->status(),
                'body_excerpt' => Str::limit($response->body(), 200),
            ]);
        }
        return $results;
    }

    protected function fetchCitiesFromApi($provinceId = null): array
    {
        if (!$this->isConfigured()) return [];
        $params = $provinceId ? ['province' => $provinceId] : [];
        try {
            $response = $this->client(['key' => $this->apiKey])->timeout(5)->get("{$this->baseUrl}/city", $params);
        } catch (\Throwable $e) {
            Log::warning('RajaOngkir cities unreachable', ['message' => $e->getMessage(), 'province_id' => $provinceId]);
            return [];
        }
        $results = $this->extractResults($response);
        if (empty($results)) {
            Log::warning('RajaOngkir cities returned empty/invalid payload', [
                'status' => $response->status(),
                'body_excerpt' => Str::limit($response->body(), 200),
                'province_id' => $provinceId,
            ]);
        }
        return $results;
    }

    protected function loadOfflineProvinces(): array
    {
        $path = resource_path('data/rajaongkir_provinces.json');
        if (!File::exists($path)) return [];
        return Cache::remember('rajaongkir_offline_provinces', now()->addDays(30), function () use ($path) {
            $raw = json_decode(File::get($path), true) ?? [];
            return array_map(function ($p) {
                return [
                    'province_id' => (string) ($p['id'] ?? ''),
                    'province' => ucwords(strtolower($p['name'] ?? '')),
                    'source' => 'offline-cache',
                ];
            }, $raw);
        });
    }

    protected function loadOfflineCities(): array
    {
        $path = resource_path('data/rajaongkir_cities.json');
        if (!File::exists($path)) return [];
        return Cache::remember('rajaongkir_offline_cities', now()->addDays(30), function () use ($path) {
            return json_decode(File::get($path), true) ?? [];
        });
    }

    protected function extractResults(Response $response): array
    {
        if (!$response->successful()) return [];
        if (!$this->responseLooksJson($response)) return [];
        $data = $response->json();
        return $data['rajaongkir']['results'] ?? [];
    }

    protected function responseLooksJson(Response $response): bool
    {
        $contentType = strtolower($response->header('content-type', ''));
        if (Str::contains($contentType, 'json')) {
            return true;
        }
        // Some gateways answer text/plain but body is JSON
        $body = trim($response->body());
        return Str::startsWith($body, '{') || Str::startsWith($body, '[');
    }

    protected function client(array $headers = [])
    {
        $req = Http::withHeaders($headers);
        if ($this->skipSslVerify) {
            $req = $req->withoutVerifying();
        }
        return $req;
    }

    protected function loadOfflineCosts($origin, $destination, $weight, $courier): array
    {
        try {
            $path = resource_path('data/rajaongkir_cost_fallback.json');
            if (!File::exists($path)) return [];

            $data = json_decode(File::get($path), true);
            if (!is_array($data)) return [];

            $key = strtolower((string)$origin) . '-' . strtolower((string)$destination) . '-' . strtolower((string)$courier);
            $rows = $data['pairs'][$key] ?? [];
            if (empty($rows)) return [];

            $multiplier = max(1, (int) ceil(((int) $weight) / 1000));

            return array_map(function ($r) use ($multiplier) {
                return [
                    'courier_code' => $r['courier_code'] ?? ($r['courier'] ?? ''),
                    'courier_name' => $r['courier_name'] ?? '',
                    'service' => $r['service'] ?? 'REG',
                    'description' => $r['description'] ?? '',
                    'cost' => ((int) ($r['cost'] ?? 0)) * $multiplier,
                    'etd' => $r['etd'] ?? '-',
                    'note' => ($r['note'] ?? '') . ' (offline-fallback)',
                ];
            }, $rows);
        } catch (\Throwable $e) {
            return [];
        }
    }

    public function getOfflineCosts($origin, $destination, $weight, $courier): array
    {
        if (!$this->offlineCostEnabled) {
            return [];
        }
        return $this->loadOfflineCosts($origin, $destination, $weight, $courier);
    }

    public function getDefaultOfflineCosts(string $courier, int $weight): array
    {
        $samples = [
            'jne' => [
                ['service' => 'REG', 'cost' => 12000, 'etd' => '1-2', 'description' => 'Regular Service Standalone'],
                ['service' => 'YES', 'cost' => 20000, 'etd' => '1', 'description' => 'Yes Express'],
            ],
            'pos' => [
                ['service' => 'REG', 'cost' => 14000, 'etd' => '2-3', 'description' => 'POS Reguler'],
            ],
            'tiki' => [
                ['service' => 'REG', 'cost' => 12500, 'etd' => '1-2', 'description' => 'TIKI Reguler'],
            ],
        ];

        $rows = [];
        $multiplier = max(1, (int) ceil($weight / 1000));
        foreach ($samples[$courier] ?? $samples['jne'] as $sample) {
            $rows[] = [
                'courier_code' => $courier,
                'courier_name' => strtoupper($courier),
                'service' => $sample['service'],
                'description' => $sample['description'] ?? '',
                'cost' => ((int) ($sample['cost'] ?? 0)) * $multiplier,
                'etd' => $sample['etd'] ?? '-',
                'note' => 'Fallback default',
            ];
        }

        return $rows;
    }
}


