<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Models\WishlistItem;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class Product extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $fillable = [
        'category_id',
        'sku',
        'name',
        'slug',
        'tagline',
        'thumbnail',
        'summary',
        'description',
        'stock',
        'min_order_qty',
        'max_order_qty',
        'weight_grams',
        'price',
        'cost_price',
        'discount_percent',
        'is_featured',
        'status',
        'metadata',
        'created_by',
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'metadata' => 'array',
    ];

    protected static function booted(): void
    {
        static::creating(function (Product $product) {
            if (empty($product->slug)) {
                $product->slug = Str::slug($product->name . '-' . Str::random(4));
            }
        });
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function images()
    {
        return $this->hasMany(ProductImage::class)->orderBy('display_order');
    }

    public function wishlistItems()
    {
        return $this->hasMany(WishlistItem::class);
    }

    public function primaryImage()
    {
        return $this->images()->where('is_primary', true)->first();
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class)->approved()->recent();
    }

    public function allReviews()
    {
        return $this->hasMany(Review::class);
    }

    public function averageRating()
    {
        $avg = $this->allReviews()->avg('rating');
        return $avg ? (float) $avg : 0.0;
    }

    public function totalReviews()
    {
        return $this->reviews()->count();
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function getFinalPriceAttribute(): int
    {
        if ($this->discount_percent > 0) {
            return (int) round($this->price * (1 - ($this->discount_percent / 100)));
        }

        return (int) $this->price;
    }

    public function getFormattedPriceAttribute(): string
    {
        return number_format($this->final_price, 0, ',', '.');
    }

    public function getThumbnailAttribute($value): ?string
    {
        // Jika value sudah null, return null
        if ($value === null || $value === '') {
            return null;
        }

        // Jika sudah URL lengkap, return langsung
        if (Str::startsWith($value, ['http://', 'https://', 'data:'])) {
            return $value;
        }

        // Gunakan URL relatif atau lengkap berdasarkan request
        $baseUrl = request()->getSchemeAndHttpHost();
        return $baseUrl . '/storage/' . $value;
    }

    public function getThumbnailUrlAttribute(): string
    {
        $rawThumbnail = $this->getRawOriginal('thumbnail');

        if ($rawThumbnail) {
            return $this->getAttribute('thumbnail');
        }

        $placeholderPath = public_path('images/placeholders/product.svg');
        if (file_exists($placeholderPath)) {
            return asset('images/placeholders/product.svg');
        }

        return 'https://via.placeholder.com/320x320.png?text=Produk';
    }
}
