<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Category extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'tagline',
        'description',
        'hero_image',
        'carousel_image',
        'display_order',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    protected static function booted(): void
    {
        static::creating(function (Category $category) {
            if (empty($category->slug)) {
                $category->slug = Str::slug($category->name);
            }
        });
    }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    public function activeProducts()
    {
        return $this->hasMany(Product::class)->where('status', 'active');
    }

    public function featuredProducts()
    {
        return $this->products()->where('is_featured', true);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)->orderBy('display_order');
    }

    public function getHeroImageUrlAttribute(): ?string
    {
        $hero = $this->getRawOriginal('hero_image');
        $carousel = $this->getRawOriginal('carousel_image');

        return $this->resolveImageUrl($hero) ?? $this->resolveImageUrl($carousel);
    }

    public function getCarouselImageUrlAttribute(): ?string
    {
        return $this->resolveImageUrl($this->getRawOriginal('carousel_image'));
    }

    protected function resolveImageUrl(?string $value): ?string
    {
        if (empty($value)) {
            return null;
        }

        if (Str::startsWith($value, ['http://', 'https://', 'data:', '//'])) {
            return $value;
        }

        if (Str::startsWith($value, ['/'])) {
            return url($value);
        }

        $storagePath = public_path('storage/' . ltrim($value, '/'));
        if (file_exists($storagePath)) {
            return url('storage/' . ltrim($value, '/'));
        }

        $publicPath = public_path($value);
        if (file_exists($publicPath)) {
            return asset($value);
        }

        return url('storage/' . ltrim($value, '/'));
    }
}
