<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class SwitchSessionCookie
{
    /**
     * Set session cookie name based on route prefix so multiple roles can stay logged in separate tabs.
     * This enables simultaneous admin/cashier/user login in different browser tabs.
     */
    public function handle(Request $request, Closure $next): Response
    {
        $path = '/' . ltrim($request->path(), '/');
        $referer = $request->headers->get('referer', '');
        $host = $request->getHost();

        // Determine session cookie based on path, referer, or host
        $sessionCookie = $this->determineSessionCookie($path, $referer, $host);
        config(['session.cookie' => $sessionCookie]);

        return $next($request);
    }

    /**
     * Determine the appropriate session cookie name based on the request context.
     */
    protected function determineSessionCookie(string $path, string $referer, string $host): string
    {
        // Admin routes
        if ($this->isAdminContext($path, $referer, $host)) {
            return 'admin_session';
        }

        // Cashier routes
        if ($this->isCashierContext($path, $referer, $host)) {
            return 'cashier_session';
        }

        // Default user session
        return config('session.cookie', 'laravel_session');
    }

    /**
     * Check if the request is in admin context.
     */
    protected function isAdminContext(string $path, string $referer, string $host): bool
    {
        // Check path patterns for admin
        $adminPaths = ['/admin', '/admin/'];
        foreach ($adminPaths as $adminPath) {
            if (str_starts_with($path, $adminPath)) {
                return true;
            }
        }

        // Check host for admin subdomain
        if (str_contains($host, 'admin.')) {
            return true;
        }

        // Check logout from admin (via referer)
        if ($path === '/logout' && str_contains($referer, '/admin')) {
            return true;
        }

        return false;
    }

    /**
     * Check if the request is in cashier context.
     */
    protected function isCashierContext(string $path, string $referer, string $host): bool
    {
        // Check path patterns for cashier
        $cashierPaths = ['/kasir', '/kasir/'];
        foreach ($cashierPaths as $cashierPath) {
            if (str_starts_with($path, $cashierPath)) {
                return true;
            }
        }

        // Check host for cashier subdomain
        if (str_contains($host, 'kasir.')) {
            return true;
        }

        // Check logout from cashier (via referer)
        if ($path === '/logout' && str_contains($referer, '/kasir')) {
            return true;
        }

        return false;
    }
}

