<?php

namespace App\Http\Middleware;

use App\Enums\UserRole;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class RoleMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string ...$roles): Response
    {
        $user = $request->user();

        if (! $user) {
            abort(Response::HTTP_FORBIDDEN, 'Silakan masuk untuk mengakses halaman ini.');
        }

        if (empty($roles)) {
            return $next($request);
        }

        $allowedRoles = collect($roles)
            ->map(fn ($role) => trim(strtolower($role)))
            ->filter()
            ->map(function (string $role) {
                return match ($role) {
                    'admin' => UserRole::Admin->value,
                    'kasir', 'cashier' => UserRole::Cashier->value,
                    'user', 'customer', 'pelanggan' => UserRole::Customer->value,
                    default => $role,
                };
            })
            ->unique()
            ->all();

        $currentRole = $user->role instanceof UserRole ? $user->role->value : $user->role;

        if (! in_array($currentRole, $allowedRoles, true)) {
            abort(Response::HTTP_FORBIDDEN, 'Anda tidak memiliki akses ke halaman ini.');
        }

        return $next($request);
    }
}
