<?php

namespace App\Http\Controllers;

use App\Services\Shipping\RajaOngkirService;
use App\Services\Shipping\ShipperService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class ShippingController extends Controller
{
    public function __construct(
        protected RajaOngkirService $rajaOngkir,
        protected ShipperService $shipper
    ) {
    }

    /**
     * Get provinces
     */
    public function provinces()
    {
        if (!$this->rajaOngkir->isConfigured()) {
            return response()->json([
                'success' => false,
                'message' => 'RajaOngkir belum dikonfigurasi',
                'data' => [],
            ], 400);
        }

        $provinces = $this->rajaOngkir->getProvinces();

        return response()->json([
            'success' => true,
            'data' => $provinces,
        ]);
    }

    /**
     * Get cities by province
     */
    public function cities(Request $request)
    {
        if (!$this->rajaOngkir->isConfigured()) {
            return response()->json([
                'success' => false,
                'message' => 'RajaOngkir belum dikonfigurasi',
                'data' => [],
            ], 400);
        }

        $provinceId = $request->query('province_id');
        $cities = $this->rajaOngkir->getCities($provinceId);

        return response()->json([
            'success' => true,
            'data' => $cities,
        ]);
    }

    /**
     * Calculate shipping cost
     */
        /**
     * Calculate shipping cost
     */
    public function calculateCost(Request $request)
    {
        $validated = $request->validate([
            'destination' => 'required', // allow string/number; we'll normalize below
            'destination_name' => 'nullable|string',
            'province' => 'nullable|string',
            'weight' => 'required|integer|min:1',
            'courier' => 'nullable|string',
        ]);

        $weight = (int) $validated['weight'];
        $destinationCityId = (int) $validated['destination'];

        // Normalize offline city_id to RajaOngkir city_id when necessary
        if ($destinationCityId > 1000 && $this->rajaOngkir->isConfigured()) {
            try {
                // If client sends destination_name like "Kabupaten Tangerang" and province
                $name = (string) ($validated['destination_name'] ?? '');
                $province = (string) ($validated['province'] ?? '');
                $type = null;
                $cityName = trim($name);
                if (stripos($name, 'Kabupaten ') === 0) { $type = 'Kabupaten'; $cityName = trim(substr($name, strlen('Kabupaten '))); }
                if (stripos($name, 'Kota ') === 0) { $type = 'Kota'; $cityName = trim(substr($name, strlen('Kota '))); }

                // Fetch live cities (bypass cache by hitting service then filtering)
                $cities = Http::withHeaders(['key' => config('services.rajaongkir.api_key')])
                    ->get((config('services.rajaongkir.base_url') ?: 'https://api.rajaongkir.com/starter') . '/city')
                    ->json('rajaongkir.results') ?? [];

                $match = collect($cities)->first(function ($c) use ($cityName, $province, $type) {
                    $okName = strcasecmp((string)($c['city_name'] ?? ''), $cityName) === 0;
                    $okProv = empty($province) || strcasecmp((string)($c['province'] ?? ''), $province) === 0;
                    $okType = empty($type) || strcasecmp((string)($c['type'] ?? ''), $type) === 0;
                    return $okName && $okProv && $okType;
                });

                if (!empty($match['city_id'])) {
                    $destinationCityId = (int) $match['city_id'];
                }
            } catch (\Throwable $e) {
                // ignore and keep the original id
            }
        }

        // Prefer Shipper if configured
        if ($this->shipper->isConfigured()) {
            $originAreaId = (int) (config('services.shipper.origin_area_id') ?: 0);
            if (!$originAreaId) {
                $originQuery = config('services.shipper.origin_area_query', 'Kabupaten Tangerang');
                $originAreaId = (int) ($this->shipper->findAreaIdByQuery($originQuery) ?: 0);
            }

            $city = $this->rajaOngkir->getCityById($destinationCityId);
            if ($city) {
                $fullName = trim(($city['type'] ?? '') . ' ' . ($city['city_name'] ?? ''));
                $destAreaId = (int) ($this->shipper->findAreaIdByQuery($fullName) ?: 0);
            }

            if (!empty($originAreaId ?? null) && !empty($destAreaId ?? null)) {
                $couriers = config('services.shipper.couriers', []);
                $rates = $this->shipper->getRatesByAreaId($originAreaId, $destAreaId, $weight, $couriers);
                return response()->json($rates);
            }
            // If mapping failed, fall back to manual RajaOngkir (if available) below
        }

        // Fallback to RajaOngkir (works for offline city list and when API configured)
        if (!$this->rajaOngkir->isConfigured()) {
            return response()->json([
                'success' => false,
                'message' => 'Perhitungan ongkir otomatis tidak tersedia. Admin akan mengonfirmasi ongkir.',
                'data' => [],
            ], 400);
        }

        $origin = config('services.rajaongkir.origin_city_id', 154);
        $couriers = !empty($validated['courier']) ? [$validated['courier']] : ['jne','pos','tiki'];

        $offlineEnabled = config('services.rajaongkir.offline_cost', false);
        if ($offlineEnabled) {
            $fallback = [];
            foreach ($couriers as $courier) {
                $rows = $this->rajaOngkir->getOfflineCosts($origin, $destinationCityId, $weight, $courier);
                if (empty($rows)) {
                    $rows = $this->rajaOngkir->getDefaultOfflineCosts($courier, $weight);
                }
                $fallback = array_merge($fallback, $rows);
            }
            if (!empty($fallback)) {
                return response()->json([
                    'success' => true,
                    'data' => $fallback,
                    'mode' => 'offline',
                ]);
            }
        }

        if (!empty($validated['courier'])) {
            $result = $this->rajaOngkir->getCost($origin, $destinationCityId, $weight, $validated['courier']);
        } else {
            $result = $this->rajaOngkir->getMultipleCourierCosts($origin, $destinationCityId, $weight, $couriers);
        }
        return response()->json($result);
    }

    /**
     * Search city
     */
    public function searchCity(Request $request)
    {
        if (!$this->rajaOngkir->canSearchCities()) {
            \Log::warning('City search requested but neither RajaOngkir nor offline dataset is ready.');
            return response()->json([
                'success' => false,
                'message' => 'Fitur pencarian kota tidak tersedia. Hubungi administrator.',
                'data' => [],
                'error' => 'City lookup unavailable',
                'mode' => $this->rajaOngkir->cityLookupMode(),
            ], 400);
        }

        $query = $request->query('q', '');
        
        if (strlen($query) < 3) {
            return response()->json([
                'success' => false,
                'message' => 'Minimal 3 karakter untuk pencarian',
                'data' => [],
                'mode' => $this->rajaOngkir->cityLookupMode(),
            ]);
        }

        try {
            $cities = $this->rajaOngkir->searchCity($query);

            // Check if we got any results
            if (empty($cities)) {
                \Log::info("RajaOngkir: No cities found for query: {$query}");
            }

            return response()->json([
                'success' => true,
                'data' => $cities,
                'count' => count($cities),
                'mode' => $this->rajaOngkir->cityLookupMode(),
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in searchCity: ' . $e->getMessage(), [
                'query' => $query,
                'trace' => $e->getTraceAsString(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mencari kota. Silakan coba lagi.',
                'data' => [],
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error',
                'mode' => $this->rajaOngkir->cityLookupMode(),
            ], 500);
        }
    }
}







