<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $categories = Category::active()
            ->withCount(['products' => function ($query) {
                $query->where('is_active', true);
            }])
            ->get();
        $selectedCategory = $request->query('category');
        $search = $request->query('search');

        $products = Product::active()
            ->with(['category', 'images'])
            ->when($selectedCategory, function ($query, $selectedCategory) {
                $query->whereHas('category', fn ($categoryQuery) => $categoryQuery->where('slug', $selectedCategory));
            })
            ->when($search, function ($query, $search) {
                $query->where(function ($builder) use ($search) {
                    $builder->where('name', 'like', "%{$search}%")
                        ->orWhere('summary', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->orderByDesc('is_featured')
            ->orderBy('name')
            ->paginate(12)
            ->withQueryString();

        return view('products.index', [
            'products' => $products,
            'categories' => $categories,
            'selectedCategory' => $selectedCategory,
            'search' => $search,
        ]);
    }

    public function show(Request $request, Product $product)
    {
        $product->loadMissing(['category', 'images']);

        // Load reviews dengan user
        $reviews = $product->reviews()->with('user')->paginate(10);
        $userReview = null;
        
        if (auth()->check()) {
            $userReview = $product->allReviews()->where('user_id', auth()->id())->first();
        }

        // Calculate rating stats
        try {
            $avgRating = (float) ($product->averageRating() ?? 0);
            $totalReviews = (int) ($product->totalReviews() ?? 0);
        } catch (\Exception $e) {
            $avgRating = 0.0;
            $totalReviews = 0;
        }

        $relatedProducts = Product::active()
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->with('images')
            ->take(4)
            ->get();

        $isWishlisted = false;

        $content = view('products.show', [
            'product' => $product,
            'relatedProducts' => $relatedProducts,
            'reviews' => $reviews,
            'userReview' => $userReview,
            'avgRating' => $avgRating,
            'totalReviews' => $totalReviews,
            'isWishlisted' => $isWishlisted,
        ])->render();

        return response($content)
            ->header('Cache-Control', 'no-cache, no-store, must-revalidate')
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }
}
