<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Symfony\Component\HttpFoundation\Response;

class OrderController extends Controller
{
    public function index(Request $request): View
    {
        $orders = Order::query()
            ->where('user_id', $request->user()->id)
            ->latest()
            ->paginate(10);

        return view('orders.index', [
            'orders' => $orders,
        ]);
    }

    public function show(Order $order): View
    {
        $this->authorizeOrder($order);

        $order->load(['items.product.images', 'payments', 'cashier']);

        return view('orders.show', [
            'order' => $order,
        ]);
    }

    public function print(Order $order): View
    {
        $this->authorizeOrder($order);

        $order->load(['items.product', 'payments', 'cashier']);

        return view('orders.print', [
            'order' => $order,
            'storeName' => config('app.name', 'D\'Shop'),
        ]);
    }

    /**
     * Cancel an order (user action)
     */
    public function cancel(Request $request, Order $order): RedirectResponse
    {
        $this->authorizeOrder($order);

        // Hanya bisa dibatalkan jika status pending atau awaiting_payment
        $allowedStatuses = ['pending', 'awaiting_payment'];
        if (!in_array($order->status, $allowedStatuses)) {
            return back()->with('error', 'Pesanan tidak dapat dibatalkan karena sudah diproses oleh toko.');
        }

        // Jika sudah dibayar, tidak bisa dibatalkan oleh user
        if ($order->payment_status === 'paid') {
            return back()->with('error', 'Pesanan yang sudah dibayar tidak dapat dibatalkan. Hubungi admin untuk pengajuan refund.');
        }

        // Update status pesanan
        $order->update([
            'status' => 'cancelled',
            'metadata' => array_merge($order->metadata ?? [], [
                'cancelled_at' => now()->toISOString(),
                'cancelled_by' => 'user',
                'cancelled_by_id' => Auth::id(),
                'cancellation_reason' => $request->input('reason', 'Dibatalkan oleh pembeli'),
            ]),
        ]);

        // Kembalikan stok produk
        foreach ($order->items as $item) {
            if ($item->product) {
                $item->product->increment('stock', $item->quantity);
            }
        }

        return redirect()->route('orders.index')
            ->with('success', 'Pesanan berhasil dibatalkan.');
    }

    protected function authorizeOrder(Order $order): void
    {
        // Cek user dari guard yang aktif (bisa web, admin, atau cashier)
        $user = Auth::guard('admin')->user()
             ?? Auth::guard('cashier')->user()
             ?? Auth::guard('web')->user();

        if (!$user) {
            abort(Response::HTTP_FORBIDDEN);
        }

        // Admin atau kasir boleh melihat semua pesanan
        if ($user->isAdmin() || $user->isCashier()) {
            return;
        }

        // Customer hanya boleh lihat pesanan miliknya sendiri
        // Gunakan == bukan !== karena bisa terjadi type mismatch (string vs int)
        if ((int) $order->user_id !== (int) $user->id) {
            abort(Response::HTTP_FORBIDDEN);
        }
    }
}
