<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function __invoke(Request $request)
    {
        $user = $request->user();

        if ($user->isAdmin()) {
            return redirect()->route('admin.dashboard');
        }

        if ($user->isCashier()) {
            return redirect()->route('cashier.dashboard');
        }

        $ordersQuery = Order::query()->where('user_id', $user->id);

        $stats = [
            'totalOrders' => (clone $ordersQuery)->count(),
            'completedOrders' => (clone $ordersQuery)->where('status', 'completed')->count(),
            'waitingPayment' => (clone $ordersQuery)->where('payment_status', '!=', 'paid')->count(),
            'totalSpend' => (clone $ordersQuery)->where('payment_status', 'paid')->sum('grand_total'),
        ];

        $insights = [
            'ordersThisMonth' => (clone $ordersQuery)
                ->whereBetween('created_at', [now()->startOfMonth(), now()->endOfMonth()])
                ->count(),
            'spendThisMonth' => (clone $ordersQuery)
                ->whereBetween('paid_at', [now()->startOfMonth(), now()->endOfMonth()])
                ->sum('grand_total'),
            'averageOrderValue' => $stats['totalOrders'] > 0
                ? (int) round((clone $ordersQuery)->average('grand_total'))
                : 0,
            'pendingShipment' => (clone $ordersQuery)
                ->whereIn('status', ['processing', 'ready_to_ship', 'shipping', 'packed'])
                ->count(),
        ];

        $nextOrder = (clone $ordersQuery)
            ->whereIn('status', ['pending', 'awaiting_payment', 'processing', 'ready_to_ship', 'shipping'])
            ->with(['items.product'])
            ->orderBy('created_at')
            ->first();

        $favoriteCategories = DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->where('orders.user_id', $user->id)
            ->select(
                'categories.id',
                'categories.name',
                DB::raw('COUNT(order_items.id) as total_items'),
                DB::raw('SUM(order_items.subtotal) as total_spent')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('total_items')
            ->limit(3)
            ->get();

        $recommendedProducts = Product::active()
            ->with('category')
            ->when($favoriteCategories->isNotEmpty(), function ($query) use ($favoriteCategories) {
                $query->whereIn('category_id', $favoriteCategories->pluck('id'));
            })
            ->latest('updated_at')
            ->limit(6)
            ->get();

        if ($recommendedProducts->isEmpty()) {
            $recommendedProducts = Product::active()
                ->with('category')
                ->latest('created_at')
                ->limit(6)
                ->get();
        }

        $activityFeed = (clone $ordersQuery)
            ->latest()
            ->with(['payments', 'items'])
            ->take(6)
            ->get();

        $savedAddresses = $user->addresses()
            ->orderByDesc('is_default')
            ->limit(3)
            ->get();

        $recentOrders = (clone $ordersQuery)->latest()->take(5)->with('payments')->get();

        // Get notification counts for dashboard
        $unpaidOrdersCount = (clone $ordersQuery)->where('payment_status', '!=', 'paid')->count();
        $pendingOrdersCount = (clone $ordersQuery)
            ->whereIn('status', ['pending', 'confirmed', 'processing', 'ready_to_ship', 'shipping'])
            ->count();

        return view('dashboard', [
            'stats' => $stats,
            'insights' => $insights,
            'recentOrders' => $recentOrders,
            'nextOrder' => $nextOrder,
            'favoriteCategories' => $favoriteCategories,
            'recommendedProducts' => $recommendedProducts,
            'activityFeed' => $activityFeed,
            'savedAddresses' => $savedAddresses,
            'unpaidOrdersCount' => $unpaidOrdersCount,
            'pendingOrdersCount' => $pendingOrdersCount,
        ]);
    }
}

