<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Payment;
use App\Models\StoreSetting;
use App\Models\UserAddress;
use App\Services\Cart\CartService;
use App\Services\Payment\PaymentGatewayService;
use App\Services\Shipping\RajaOngkirService;
use App\Services\Shipping\ShipperService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CheckoutController extends Controller
{
    public function __construct(
        protected CartService $cartService,
        protected PaymentGatewayService $paymentGateway,
        protected RajaOngkirService $rajaOngkir,
        protected ShipperService $shipper
    ) {
    }

    public function index(Request $request)
    {
        $cart = $this->cartService->resolve($request);
        $shippingEstimatorEnabled = $this->shipper->isConfigured() || $this->rajaOngkir->isConfigured();

        if (! $cart) {
            return redirect()->route('cart.index')->with('status', 'Keranjang masih kosong.');
        }

        $items = $this->cartService->selectedItems($cart);

        if ($items->isEmpty()) {
            return redirect()->route('cart.index')->with('status', 'Silakan pilih produk yang ingin di checkout.');
        }

        $defaultAddress = $request->user()?->addresses()->where('is_default', true)->first();

        $paymentMethods = [
            ['code' => 'midtrans', 'label' => 'Virtual Account / QRIS (Midtrans)'],
            ['code' => 'manual_transfer', 'label' => 'Transfer Manual'],
        ];

        // Calculate total weight for shipping
        $totalWeight = $items->sum(function ($item) {
            return ($item->product->weight_grams ?? 0) * $item->quantity;
        });

        $cityLookupMode = $this->rajaOngkir->cityLookupMode();
        $citySearchEnabled = $cityLookupMode !== 'disabled';
        $shippingEstimatorEnabled = $this->shipper->isConfigured() || $this->rajaOngkir->isConfigured();
        $shippingProviderLabel = $this->shipper->isConfigured()
            ? 'Shipper'
            : ($this->rajaOngkir->isConfigured() ? 'RajaOngkir' : 'Manual');

        return view('checkout.index', [
            'cart' => $cart,
            'items' => $items,
            'defaultAddress' => $defaultAddress,
            'paymentMethods' => $paymentMethods,
            'adminWhatsapp' => StoreSetting::getValue('store.whatsapp_admin', '6283879865012'),
            'totalWeight' => $totalWeight,
            'rajaOngkirEnabled' => $shippingEstimatorEnabled,
            'citySearchEnabled' => $citySearchEnabled,
            'shippingProviderLabel' => $shippingProviderLabel,
            'selectedShipping' => [],
            'cityLookupMode' => $cityLookupMode,
        ]);
    }

    public function store(Request $request)
    {
        $cart = $this->cartService->resolve($request);

        if (! $cart) {
            return redirect()->route('cart.index')->withErrors('Keranjang tidak ditemukan.');
        }

        $items = $this->cartService->selectedItems($cart);

        if ($items->isEmpty()) {
            return redirect()->route('cart.index')->withErrors('Silakan pilih produk yang ingin dibayar.');
        }

        $cityLookupMode = $this->rajaOngkir->cityLookupMode();
        $cityIdRule = $cityLookupMode === 'disabled' ? ['nullable', 'integer'] : ['required', 'integer'];

        $validated = $request->validate([
            'recipient_name' => ['required', 'string', 'max:120'],
            'phone' => ['required', 'string', 'max:25'],
            'address_line1' => ['required', 'string', 'max:255'],
            'address_line2' => ['nullable', 'string', 'max:255'],
            'city' => ['required', 'string', 'max:120'],
            'city_id' => $cityIdRule, // RajaOngkir city ID or offline fallback ID
            'province' => ['required', 'string', 'max:120'],
            'postal_code' => ['nullable', 'string', 'max:15'],
            'payment_method' => ['required', 'in:midtrans,manual_transfer,cash'],
            'shipping_courier' => ['nullable', 'string'],
            'shipping_service' => ['nullable', 'string'],
            'shipping_cost' => ['nullable', 'integer', 'min:0'],
            'notes' => ['nullable', 'string'],
            'confirm_payment_responsibility' => ['accepted'],
            'confirm_shipping_choice' => ['accepted'],
            'confirm_online_receipt' => ['accepted'],
        ]);

        $shippingCost = (int) ($validated['shipping_cost'] ?? 0);

        $order = DB::transaction(function () use ($validated, $items, $shippingCost, $request, $cart) {
            $subtotal = $items->sum(fn ($item) => $item->subtotal);
            $discount = (int) round($subtotal * 0.05);
            $grandTotal = max($subtotal - $discount + $shippingCost, 0);

            $order = Order::create([
                'user_id' => $request->user()?->id,
                'cashier_id' => null,
                'cart_id' => $cart->id,
                'channel' => 'online',
                'status' => 'pending',
                'payment_status' => 'unpaid',
                'payment_method' => $validated['payment_method'],
                'payment_gateway' => $validated['payment_method'] === 'midtrans' ? 'midtrans' : 'manual',
                'payment_reference' => strtoupper(Str::random(12)),
                'payment_charge_party' => 'buyer',
                'total_items' => $items->sum('quantity'),
                'subtotal_amount' => $subtotal,
                'discount_amount' => $discount,
                'shipping_cost' => $shippingCost,
                'service_fee' => 0,
                'grand_total' => $grandTotal,
                'shipping_address' => [
                    'recipient_name' => $validated['recipient_name'],
                    'phone' => $validated['phone'],
                    'address_line1' => $validated['address_line1'],
                    'address_line2' => $validated['address_line2'] ?? null,
                    'city' => $validated['city'],
                    'city_id' => $validated['city_id'] ?? null,
                    'province' => $validated['province'],
                    'postal_code' => $validated['postal_code'] ?? null,
                ],
                'shipping_courier' => $validated['shipping_courier'] ?? null,
                'expedition_preference' => null,
                'shipping_service' => $validated['shipping_service'] ?? null,
                'tracking_number' => null,
                'notes' => $validated['notes'] ?? null,
                'payment_due_at' => now()->addDay(),
                'metadata' => [
                    'source' => 'website',
                    'purchase_agreements' => [
                        'payment_fee_by_buyer' => $request->boolean('confirm_payment_responsibility'),
                        'shipping_selection_acknowledged' => $request->boolean('confirm_shipping_choice'),
                        'online_receipt_acknowledged' => $request->boolean('confirm_online_receipt'),
                    ],
                ],
            ]);

            $items->each(function ($item) use ($order) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'product_name' => $item->product->name,
                    'sku' => $item->product->sku,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'discount_percent' => $item->product->discount_percent,
                    'subtotal' => $item->subtotal,
                    'metadata' => ['category' => $item->product->category->name],
                ]);

                $item->product->decrement('stock', $item->quantity);
            });

            if ($validated['payment_method'] === 'cash') {
                $order->update([
                    'status' => 'confirmed',
                    'payment_status' => 'awaiting_confirmation',
                ]);
            }

            return $order;
        });

        $payment = match ($validated['payment_method']) {
            'midtrans' => $this->paymentGateway->createSnapToken($order),
            'manual_transfer' => Payment::create([
                'order_id' => $order->id,
                'amount' => $order->grand_total,
                'method' => 'manual_transfer',
                'gateway' => 'manual',
                'status' => 'waiting_confirmation',
                'reference' => 'MAN' . strtoupper(Str::random(10)),
                'notes' => 'Silakan konfirmasi pembayaran melalui WhatsApp admin.',
            ]),
            default => Payment::create([
                'order_id' => $order->id,
                'amount' => $order->grand_total,
                'method' => 'cash',
                'gateway' => 'manual',
                'status' => 'waiting_confirmation',
                'reference' => 'CASH' . strtoupper(Str::random(8)),
                'notes' => 'Pembayaran dilakukan saat mengambil barang di toko.',
            ]),
        };

        $this->cartService->clearSelectedItems($cart);

        if ($request->boolean('save_address') && $request->user()) {
            UserAddress::updateOrCreate(
                ['user_id' => $request->user()->id, 'is_default' => true],
                [
                    'label' => 'Utama',
                    'recipient_name' => $validated['recipient_name'],
                    'phone' => $validated['phone'],
                    'address_line1' => $validated['address_line1'],
                    'address_line2' => $validated['address_line2'] ?? null,
                    'city' => $validated['city'],
                    'province' => $validated['province'],
                    'postal_code' => $validated['postal_code'] ?? null,
                    'is_default' => true,
                ]
            );
        }

        // Redirect ke payment page jika menggunakan Midtrans
        if ($validated['payment_method'] === 'midtrans' && $payment->snap_token) {
            return redirect()
                ->route('orders.payment', $order)
                ->with([
                    'status' => 'Pesanan berhasil dibuat. Silakan selesaikan pembayaran.',
                ]);
        }

        return redirect()
            ->route('orders.show', $order)
            ->with([
                'status' => 'Pesanan berhasil dibuat.',
                'payment_reference' => $payment->reference,
            ]);
    }

    public function payment(Order $order)
    {
        $this->authorizePaymentView($order);

        $payment = $order->payments()
            ->where('gateway', 'midtrans')
            ->latest()
            ->first();

        if (!$payment || !$payment->snap_token) {
            return redirect()
                ->route('orders.show', $order)
                ->withErrors('Token pembayaran tidak ditemukan.');
        }

        // Check if snap token is simulated (UUID format that was manually created)
        // Note: Midtrans tokens ARE UUIDs, so we need to check if payment was created by createSimulatedPayment
        // We can check the payload to see if it contains the simulated message
        $isSimulatedToken = false;
        if ($payment->payload && isset($payment->payload['message']) && str_contains($payment->payload['message'], 'Simulated')) {
            $isSimulatedToken = true;
        }
        
        // If simulated token but Midtrans is configured, regenerate token
        if ($isSimulatedToken && !empty(config('services.midtrans.server_key'))) {
            try {
                // Create new payment with real token instead of deleting old one
                $newPayment = $this->paymentGateway->createSnapToken($order);
                
                // Update order to use new payment
                $order->payments()->where('gateway', 'midtrans')
                    ->where('id', '!=', $newPayment->id)
                    ->update(['status' => 'failed']);
                
                $payment = $newPayment;
            } catch (\Exception $e) {
                \Log::error('Failed to regenerate Midtrans token: ' . $e->getMessage());
                return redirect()
                    ->route('orders.show', $order)
                    ->withErrors('Gagal membuat token pembayaran: ' . $e->getMessage());
            }
        }

        // Validate client key
        $clientKey = config('services.midtrans.client_key');
        if (empty($clientKey)) {
            return redirect()
                ->route('orders.show', $order)
                ->withErrors('Konfigurasi Midtrans belum lengkap. Silakan hubungi administrator.');
        }

        return view('orders.payment', [
            'order' => $order,
            'payment' => $payment->fresh(),
            'midtransClientKey' => $clientKey,
            'isProduction' => config('services.midtrans.is_production', false),
        ]);
    }

    private function authorizePaymentView(Order $order): void
    {
        $admin = auth('admin')->user();
        $cashier = auth('cashier')->user();
        $customer = auth('web')->user();

        // Admin/kasir boleh mengakses semua order
        if ($admin?->isAdmin() || $cashier?->isCashier() || $cashier?->isAdmin()) {
            return;
        }

        // Jika order tidak punya user_id (guest checkout), izinkan akses
        if (empty($order->user_id)) {
            return;
        }

        // Customer yang login boleh mengakses (untuk fix session issue di hosting)
        if ($customer) {
            return;
        }

        // Jika tidak ada yang login, redirect ke login
        abort(403, 'Silakan login untuk melanjutkan pembayaran.');
    }
}










