<?php

namespace App\Http\Controllers\Cashier;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Payment;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\View\View;
use Symfony\Component\HttpFoundation\Response;

class CashierOrderController extends Controller
{
    public function create()
    {
        return redirect()->route('cashier.dashboard');
    }

    public function store(Request $request)
    {
        $isAjax = $request->wantsJson() || $request->ajax();
        
        $filteredItems = collect($request->input('items', []))
            ->filter(fn ($item) => ! empty($item['product_id']) && ! empty($item['quantity']))
            ->map(fn ($item) => [
                'product_id' => $item['product_id'],
                'quantity' => (int) $item['quantity'],
            ])
            ->values()
            ->all();

        if (empty($filteredItems)) {
            if ($isAjax) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Pilih minimal satu produk untuk dicatat.',
                    'errors' => ['items' => ['Pilih minimal satu produk untuk dicatat.']]
                ], 422);
            }
            return back()->withErrors(['items' => 'Pilih minimal satu produk untuk dicatat.'])->withInput();
        }

        // Untuk transaksi kasir, kita paksa status dibayar dan cetak struk
        $request->merge([
            'items' => $filteredItems,
            'payment_status' => 'paid',
            'print' => true,
        ]);

        // Validasi dengan try-catch untuk AJAX
        try {
            $validated = $request->validate([
                'customer_name' => ['required', 'string', 'max:120'],
                'customer_phone' => ['nullable', 'string', 'max:25'],
                'items' => ['required', 'array', 'min:1'],
                'items.*.product_id' => ['required', 'exists:products,id'],
                'items.*.quantity' => ['required', 'integer', 'min:1'],
                'notes' => ['nullable', 'string'],
                'payment_status' => ['required', 'in:paid,waiting_confirmation'],
                'discount_amount' => ['nullable', 'integer', 'min:0'],
                'print' => ['nullable', 'boolean'],
                'cash_received' => ['nullable', 'integer', 'min:0'],
                'cash_change' => ['nullable', 'integer', 'min:0'],
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($isAjax) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validasi gagal',
                    'errors' => $e->errors()
                ], 422);
            }
            throw $e;
        }

        $cashier = Auth::user();

        $order = DB::transaction(function () use ($validated, $cashier, $request) {
            $subtotal = 0;
            $totalItems = 0;
            $cashReceived = (int) ($request->input('cash_received', 0));
            $cashChange = (int) ($request->input('cash_change', 0));
            $discountAmount = (int) ($request->input('discount_amount', 0));
            $order = Order::create([
                'user_id' => null,
                'cashier_id' => $cashier?->id,
                'channel' => 'offline',
                'status' => 'completed',
                'payment_status' => 'paid',
                'payment_method' => 'cash',
                'payment_gateway' => 'manual',
                'payment_reference' => 'POS' . strtoupper(Str::random(8)),
                'payment_charge_party' => 'buyer',
                'total_items' => 0,
                'subtotal_amount' => 0,
                'discount_amount' => $discountAmount,
                'shipping_cost' => 0,
                'service_fee' => 0,
                'grand_total' => 0,
                'shipping_address' => [
                    'recipient_name' => $validated['customer_name'],
                    'phone' => $validated['customer_phone'] ?? null,
                    'address_line1' => 'Transaksi langsung di toko',
                    'city' => 'Offline Store',
                    'province' => 'Offline Store',
                ],
                'shipping_courier' => 'pickup',
                'expedition_preference' => 'pickup',
                'shipping_service' => 'pickup',
                'tracking_number' => null,
                'notes' => $validated['notes'] ?? null,
                'payment_due_at' => now(),
                'paid_at' => now(),
                'fulfilled_at' => now(),
                'metadata' => [
                    'customer_name' => $validated['customer_name'],
                    'cash_received' => $cashReceived,
                    'cash_change' => $cashChange,
                ],
            ]);

            foreach ($validated['items'] as $item) {
                $product = Product::findOrFail($item['product_id']);
                $quantity = $item['quantity'];
                $lineSubtotal = $product->final_price * $quantity;

                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'sku' => $product->sku,
                    'quantity' => $quantity,
                    'unit_price' => $product->final_price,
                    'discount_percent' => $product->discount_percent,
                    'subtotal' => $lineSubtotal,
                    'metadata' => ['origin' => 'cashier_pos'],
                ]);

                $subtotal += $lineSubtotal;
                $totalItems += $quantity;

                $product->decrement('stock', $quantity);
            }

            $grandTotal = max($subtotal - $discountAmount, 0);
            
            $order->update([
                'total_items' => $totalItems,
                'subtotal_amount' => $subtotal,
                'discount_amount' => $discountAmount,
                'grand_total' => $grandTotal,
            ]);

            $payment = Payment::create([
                'order_id' => $order->id,
                'amount' => $grandTotal,
                'method' => 'cash',
                'gateway' => 'manual',
                'status' => 'paid',
                'reference' => 'CASH' . strtoupper(Str::random(8)),
                'notes' => 'Transaksi dari kasir offline.',
                'paid_at' => now(),
                'payload' => [
                    'cash_received' => $cashReceived,
                    'cash_change' => $cashChange,
                ],
            ]);

            // Send email notification if paid
            if ($order->user && $order->user->email) {
                try {
                    \Illuminate\Support\Facades\Mail::to($order->user->email)
                        ->send(new \App\Mail\PaymentSuccessNotification($order->fresh()));
                } catch (\Exception $e) {
                    \Log::error('Failed to send payment success email from cashier: ' . $e->getMessage(), [
                        'order_id' => $order->id,
                        'email' => $order->user->email,
                    ]);
                }
            }

            return $order;
        });

        // Generate print URL dengan auto print
        $printUrl = route('cashier.orders.print', ['order' => $order->id, 'auto_print' => 1]);

        // Untuk AJAX request, return JSON
        if ($isAjax) {
            return response()->json([
                'status' => 'success',
                'message' => 'Transaksi berhasil!',
                'order_id' => $order->id,
                'invoice_number' => $order->invoice_number,
                'print_url' => $printUrl,
            ], 200);
        }

        // Untuk non-AJAX (fallback), redirect biasa
        return redirect($printUrl)->with('status', 'Transaksi berhasil! Struk akan dicetak otomatis.');
    }

    public function show(Order $order): View
    {
        $this->authorizeOrder($order);

        $order->load(['items.product', 'payments', 'cashier']);

        return view('cashier.orders.show', [
            'order' => $order,
        ]);
    }

    public function markPaid(Order $order)
    {
        $this->authorizeOrder($order);

        $order->update([
            'payment_status' => 'paid',
            'status' => 'completed',
            'paid_at' => now(),
            'fulfilled_at' => now(),
        ]);

        $payment = $order->payments()->latest()->first();

        if ($payment) {
            $payment->update([
                'status' => 'paid',
                'paid_at' => now(),
            ]);
        } else {
            Payment::create([
                'order_id' => $order->id,
                'amount' => $order->grand_total,
                'method' => 'cash',
                'gateway' => 'manual',
                'status' => 'paid',
                'reference' => 'CASH' . strtoupper(Str::random(8)),
                'paid_at' => now(),
            ]);
        }

        // Send email notification
        $order = $order->fresh();
        if ($order->user && $order->user->email) {
            try {
                \Illuminate\Support\Facades\Mail::to($order->user->email)
                    ->send(new \App\Mail\PaymentSuccessNotification($order));
            } catch (\Exception $e) {
                \Log::error('Failed to send payment success email from markPaid: ' . $e->getMessage(), [
                    'order_id' => $order->id,
                    'email' => $order->user->email,
                ]);
            }
        }

        return back()->with('status', 'Pembayaran berhasil dikonfirmasi.');
    }

    public function print(Request $request, Order $order): View
    {
        $this->authorizeOrder($order);

        $order->load(['items.product', 'payments', 'cashier']);

        // Set auto print jika ada parameter
        $autoPrint = $request->boolean('auto_print', false);

        return view('cashier.orders.print', [
            'order' => $order,
            'storeName' => config('app.name', 'D\'Shop'),
            'autoPrint' => $autoPrint,
        ]);
    }

    protected function authorizeOrder(Order $order): void
    {
        $user = Auth::user();

        if (! $user || (! $user->isCashier() && ! $user->isAdmin())) {
            abort(Response::HTTP_FORBIDDEN);
        }

        if ($order->channel !== 'offline') {
            abort(Response::HTTP_FORBIDDEN);
        }
    }
}
