<?php

namespace App\Http\Controllers;

use App\Models\CartItem;
use App\Models\Product;
use App\Services\Cart\CartService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    public function __construct(
        protected CartService $cartService
    ) {
    }

    public function index(Request $request)
    {
        $cart = $this->cartService->resolve($request);

        return view('cart.index', [
            'cart' => $cart?->load('items.product'),
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => ['required', 'exists:products,id'],
            'quantity' => ['nullable', 'integer', 'min:1'],
        ]);

        $product = Product::active()->findOrFail($request->integer('product_id'));
        $quantity = max($product->min_order_qty, (int) $request->input('quantity', 1));
        $quantity = $product->max_order_qty ? min($quantity, $product->max_order_qty) : $quantity;

        $cart = $this->cartService->resolve($request, createIfMissing: true);

        $item = $cart->items()->firstOrNew([
            'product_id' => $product->id,
        ]);

        $item->quantity = $item->exists ? $item->quantity + $quantity : $quantity;
        $item->unit_price = $product->final_price;
        $item->is_selected = true;
        $item->save();

        $cartCount = $cart->items()->sum('quantity');

        if ($request->expectsJson()) {
            return response()->json([
                'message' => "{$product->name} berhasil ditambahkan ke keranjang.",
                'cart_count' => $cartCount,
                'item' => [
                    'id' => $item->id,
                    'product_id' => $product->id,
                    'quantity' => $item->quantity,
                ],
            ]);
        }

        return redirect()
            ->route('cart.index')
            ->with('status', "{$product->name} berhasil ditambahkan ke keranjang.");
    }

    public function buyNow(Request $request)
    {
        $request->validate([
            'product_id' => ['required', 'exists:products,id'],
            'quantity' => ['nullable', 'integer', 'min:1'],
        ]);

        $product = Product::active()->findOrFail($request->integer('product_id'));
        $quantity = max($product->min_order_qty, (int) $request->input('quantity', 1));
        $quantity = $product->max_order_qty ? min($quantity, $product->max_order_qty) : $quantity;

        // Hapus semua item di cart yang tidak dipilih
        $cart = $this->cartService->resolve($request, createIfMissing: true);
        
        // Hapus semua item yang ada
        $cart->items()->delete();

        // Tambahkan produk yang ingin dibeli sekarang
        $cart->items()->create([
            'product_id' => $product->id,
            'quantity' => $quantity,
            'unit_price' => $product->final_price,
            'is_selected' => true,
        ]);

        // Redirect langsung ke checkout
        return redirect()
            ->route('checkout.index')
            ->with('status', "Lanjutkan pembayaran untuk {$product->name}.");
    }

    public function update(Request $request, CartItem $cartItem)
    {
        $this->authorizeCartItem($cartItem);

        $request->validate([
            'quantity' => ['required', 'integer', 'min:1'],
        ]);

        $product = $cartItem->product;
        $quantity = max($product->min_order_qty, $request->integer('quantity'));
        $quantity = $product->max_order_qty ? min($quantity, $product->max_order_qty) : $quantity;

        $cartItem->update([
            'quantity' => $quantity,
            'unit_price' => $product->final_price,
        ]);

        return back()->with('status', 'Keranjang berhasil diperbarui.');
    }

    public function destroy(CartItem $cartItem)
    {
        $this->authorizeCartItem($cartItem);

        $cartItem->delete();

        return back()->with('status', 'Produk dihapus dari keranjang.');
    }

    public function updateSelection(Request $request)
    {
        $cart = $this->cartService->resolve($request);

        if (! $cart) {
            return back();
        }

        $request->validate([
            'selected_items' => ['nullable', 'array'],
            'selected_items.*' => ['integer'],
        ]);

        $selected = collect($request->input('selected_items', []))->map(fn ($id) => (int) $id)->all();

        $cart->items->each(function (CartItem $item) use ($selected) {
            $item->update(['is_selected' => in_array($item->id, $selected, true)]);
        });

        return back()->with('status', 'Pilihan checkout berhasil diperbarui.');
    }

    protected function authorizeCartItem(CartItem $cartItem): void
    {
        $userId = Auth::id();
        $sessionId = session()->getId();

        if ($cartItem->cart->user_id && $cartItem->cart->user_id !== $userId) {
            abort(403);
        }

        if (is_null($cartItem->cart->user_id) && $cartItem->cart->session_id !== $sessionId) {
            abort(403);
        }
    }
}
