<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StoreSetting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Illuminate\View\View;

class AdminSettingController extends Controller
{
    public function index(): View
    {
        return view('admin.settings.index', [
            'name' => (string) (StoreSetting::getValue('store.name', config('app.name', "D'Shop")) ?? ''),
            'logo' => (string) (StoreSetting::getValue('store.logo', 'assets/logo-dshop.png') ?? ''),
            'tagline' => (string) (StoreSetting::getValue('store.tagline', 'Solusi UMKM Profesional.') ?? ''),
            'brandColor' => (string) (StoreSetting::getValue('store.brand_color', '#F7BFA0') ?? ''),
            'whatsapp' => (string) (StoreSetting::getValue('store.whatsapp_admin', '6283879865012') ?? ''),
        ]);
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:120'],
            'logo' => ['nullable', 'string', 'max:500'],
            'tagline' => ['nullable', 'string', 'max:200'],
            'brand_color' => ['nullable', 'string', 'max:20'],
            'whatsapp' => ['nullable', 'string', 'max:20'],
        ]);

        StoreSetting::updateOrCreate(['key' => 'store.name'], ['value' => (string) $data['name']]);
        if (array_key_exists('logo', $data)) {
            StoreSetting::updateOrCreate(['key' => 'store.logo'], ['value' => (string) $data['logo']]);
        }
        StoreSetting::updateOrCreate(['key' => 'store.tagline'], ['value' => (string) ($data['tagline'] ?? '')]);
        StoreSetting::updateOrCreate(['key' => 'store.brand_color'], ['value' => (string) ($data['brand_color'] ?? '#F7BFA0')]);
        StoreSetting::updateOrCreate(['key' => 'store.whatsapp_admin'], ['value' => (string) ($data['whatsapp'] ?? '')]);

        cache()->flush();

        return back()->with('status', 'Pengaturan toko diperbarui.');
    }

    /**
     * Update admin password
     */
    public function updatePassword(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'current_password' => ['required', 'string'],
            'password' => ['required', 'string', Password::min(8), 'confirmed'],
        ], [
            'current_password.required' => 'Password lama wajib diisi.',
            'password.required' => 'Password baru wajib diisi.',
            'password.min' => 'Password baru minimal 8 karakter.',
            'password.confirmed' => 'Konfirmasi password tidak cocok.',
        ]);

        $admin = Auth::guard('admin')->user();

        // Verify current password
        if (!Hash::check($validated['current_password'], $admin->password)) {
            return back()->withErrors([
                'current_password' => 'Password lama tidak sesuai.',
            ])->withInput();
        }

        // Update password
        $admin->update([
            'password' => Hash::make($validated['password']),
        ]);

        return back()->with('password_status', 'Password berhasil diperbarui!');
    }
}
