<?php

namespace App\Http\Controllers\Account;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\View\View;

class NotificationController extends Controller
{
    public function index(Request $request): View
    {
        $user = $request->user();
        
        // Get unpaid orders
        $unpaidOrders = Order::where('user_id', $user->id)
            ->where('payment_status', '!=', 'paid')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Get recent orders (last 7 days)
        $recentOrders = Order::where('user_id', $user->id)
            ->where('created_at', '>=', now()->subDays(7))
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Get orders that need attention (pending, processing)
        $pendingOrders = Order::where('user_id', $user->id)
            ->whereIn('status', ['pending', 'confirmed', 'processing', 'ready_to_ship', 'shipping'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Combine into notifications
        $notifications = collect();
        
        foreach ($unpaidOrders as $order) {
            $notifications->push([
                'type' => 'payment_required',
                'title' => 'Menunggu Pembayaran',
                'message' => "Pesanan {$order->invoice_number} menunggu pembayaran sebesar Rp" . number_format($order->grand_total, 0, ',', '.'),
                'order' => $order,
                'created_at' => $order->created_at,
                'icon' => 'credit-card',
                'color' => 'amber',
            ]);
        }
        
        foreach ($pendingOrders as $order) {
            if ($order->payment_status === 'paid') {
                $notifications->push([
                    'type' => 'order_processing',
                    'title' => 'Pesanan Diproses',
                    'message' => "Pesanan {$order->invoice_number} sedang dalam proses " . ucfirst(str_replace('_', ' ', $order->status)),
                    'order' => $order,
                    'created_at' => $order->updated_at,
                    'icon' => 'truck',
                    'color' => 'blue',
                ]);
            }
        }
        
        // Sort by date
        $notifications = $notifications->sortByDesc('created_at');
        
        return view('account.notifications.index', [
            'notifications' => $notifications,
            'unpaidCount' => $unpaidOrders->count(),
            'pendingCount' => $pendingOrders->count(),
        ]);
    }
}
