<?php

namespace App\Console\Commands;

use App\Models\CartItem;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\ProductImage;
use App\Models\Review;
use App\Models\WishlistItem;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ClearAllProducts extends Command
{
    protected $signature = 'products:clear-all';

    protected $description = 'Hapus semua produk dan data terkait (CartItem, OrderItem, ProductImage, Review, WishlistItem)';

    public function handle()
    {
        if (!$this->confirm('Apakah Anda yakin ingin menghapus SEMUA produk dan data terkait? Tindakan ini tidak dapat dibatalkan!')) {
            $this->info('Operasi dibatalkan.');
            return 0;
        }

        $this->info('Memulai penghapusan semua produk...');

        DB::transaction(function () {
            $products = Product::withTrashed()->get();
            $totalProducts = $products->count();

            $this->info("Menemukan {$totalProducts} produk untuk dihapus.");

            $products->each(function ($product) {
                if ($product->getRawOriginal('thumbnail')) {
                    $thumbnailPath = $product->getRawOriginal('thumbnail');
                    if (!is_string($thumbnailPath) || (!str_starts_with($thumbnailPath, 'http://') && !str_starts_with($thumbnailPath, 'https://') && !str_starts_with($thumbnailPath, 'data:'))) {
                        if (is_string($thumbnailPath)) {
                            Storage::disk('public')->delete($thumbnailPath);
                        }
                    }
                }

                $product->images->each(function ($image) {
                    $imagePath = $image->getRawOriginal('path');
                    if (!is_string($imagePath) || (!str_starts_with($imagePath, 'http://') && !str_starts_with($imagePath, 'https://') && !str_starts_with($imagePath, 'data:'))) {
                        if (is_string($imagePath)) {
                            Storage::disk('public')->delete($imagePath);
                        }
                    }
                    $image->delete();
                });
            });

            CartItem::query()->delete();
            $this->info('✓ CartItem dihapus.');

            OrderItem::query()->delete();
            $this->info('✓ OrderItem dihapus.');

            ProductImage::query()->delete();
            $this->info('✓ ProductImage dihapus.');

            Review::query()->delete();
            $this->info('✓ Review dihapus.');

            WishlistItem::query()->delete();
            $this->info('✓ WishlistItem dihapus.');

            Product::withTrashed()->forceDelete();
            $this->info('✓ Product dihapus.');
        });

        $this->info('✓ Semua produk dan data terkait berhasil dihapus!');
        return 0;
    }
}
