<?php

namespace App\Models;

use CodeIgniter\Model;

class ActivitiesModel extends Model
{
    protected $table = 'activities';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'judul',
        'tanggal', 
        'lokasi',
        'deskripsi',
        'media'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    // Validation
    protected $validationRules = [
        'judul' => 'required|min_length[3]|max_length[255]',
        'tanggal' => 'required|valid_date',
        'lokasi' => 'required|max_length[255]',
        'deskripsi' => 'required'
    ];

    protected $validationMessages = [
        'judul' => [
            'required' => 'Judul aktivitas harus diisi',
            'min_length' => 'Judul minimal 3 karakter',
            'max_length' => 'Judul maksimal 255 karakter'
        ],
        'tanggal' => [
            'required' => 'Tanggal aktivitas harus diisi',
            'valid_date' => 'Format tanggal tidak valid'
        ],
        'lokasi' => [
            'required' => 'Lokasi aktivitas harus diisi',
            'max_length' => 'Lokasi maksimal 255 karakter'
        ],
        'deskripsi' => [
            'required' => 'Deskripsi aktivitas harus diisi'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Get activities with search functionality
     */
    public function getActivitiesWithSearch($search = null, $limit = 10, $offset = 0)
    {
        $builder = $this->builder();
        
        if ($search) {
            $builder->groupStart()
                ->like('judul', $search)
                ->orLike('deskripsi', $search)
                ->orLike('lokasi', $search)
                ->groupEnd();
        }
        
        return $builder->orderBy('tanggal', 'DESC')
                      ->limit($limit, $offset)
                      ->get()
                      ->getResultArray();
    }

    /**
     * Get activities by date range
     */
    public function getActivitiesByDateRange($startDate, $endDate)
    {
        return $this->where('tanggal >=', $startDate)
                   ->where('tanggal <=', $endDate)
                   ->orderBy('tanggal', 'DESC')
                   ->findAll();
    }

    /**
     * Get recent activities
     */
    public function getRecentActivities($limit = 5)
    {
        return $this->orderBy('tanggal', 'DESC')
                   ->limit($limit)
                   ->findAll();
    }
}