<?php

/**
 * Admin Helper Functions
 * 
 * Helper functions untuk mempermudah pengelolaan admin
 */

if (!function_exists('is_admin_logged_in')) {
    /**
     * Check if admin is logged in
     * 
     * @return bool
     */
    function is_admin_logged_in(): bool
    {
        $session = \Config\Services::session();
        return (bool) $session->get('admin_logged_in');
    }
}

if (!function_exists('get_admin_data')) {
    /**
     * Get current admin data from session
     * 
     * @param string|null $key Specific key to get, null for all data
     * @return mixed
     */
    function get_admin_data(?string $key = null)
    {
        $session = \Config\Services::session();
        
        if (!is_admin_logged_in()) {
            return null;
        }
        
        $adminData = [
            'id' => $session->get('admin_id'),
            'username' => $session->get('admin_username'),
            'email' => $session->get('admin_email'),
            'full_name' => $session->get('admin_full_name'),
            'role' => $session->get('admin_role')
        ];
        
        if ($key) {
            return $adminData[$key] ?? null;
        }
        
        return $adminData;
    }
}

if (!function_exists('admin_has_role')) {
    /**
     * Check if admin has specific role
     * 
     * @param string|array $roles Role(s) to check
     * @return bool
     */
    function admin_has_role($roles): bool
    {
        if (!is_admin_logged_in()) {
            return false;
        }
        
        $adminRole = get_admin_data('role');
        
        if (is_array($roles)) {
            return in_array($adminRole, $roles);
        }
        
        return $adminRole === $roles;
    }
}

if (!function_exists('is_super_admin')) {
    /**
     * Check if current admin is super admin
     * 
     * @return bool
     */
    function is_super_admin(): bool
    {
        return admin_has_role('super_admin');
    }
}

if (!function_exists('admin_redirect_if_not_logged_in')) {
    /**
     * Redirect to login if admin not logged in
     * 
     * @param string|null $message Flash message
     * @return \CodeIgniter\HTTP\RedirectResponse|null
     */
    function admin_redirect_if_not_logged_in(?string $message = null)
    {
        if (!is_admin_logged_in()) {
            $message = $message ?: 'Silakan login terlebih dahulu';
            return redirect()->to('/admin/login')->with('error', $message);
        }
        
        return null;
    }
}

if (!function_exists('admin_redirect_if_not_role')) {
    /**
     * Redirect if admin doesn't have required role
     * 
     * @param string|array $requiredRoles Required role(s)
     * @param string|null $message Flash message
     * @return \CodeIgniter\HTTP\RedirectResponse|null
     */
    function admin_redirect_if_not_role($requiredRoles, ?string $message = null)
    {
        if (!admin_has_role($requiredRoles)) {
            $message = $message ?: 'Anda tidak memiliki akses untuk halaman ini';
            return redirect()->to('/admin/dashboard')->with('error', $message);
        }
        
        return null;
    }
}

if (!function_exists('format_admin_role')) {
    /**
     * Format admin role for display
     * 
     * @param string $role
     * @return string
     */
    function format_admin_role(string $role): string
    {
        $roles = [
            'super_admin' => 'Super Administrator',
            'admin' => 'Administrator',
            'moderator' => 'Moderator'
        ];
        
        return $roles[$role] ?? ucfirst(str_replace('_', ' ', $role));
    }
}

if (!function_exists('admin_menu_active')) {
    /**
     * Check if admin menu is active
     * 
     * @param string $path Menu path to check
     * @return string CSS class for active menu
     */
    function admin_menu_active(string $path): string
    {
        $currentPath = uri_string();
        
        if ($currentPath === $path || strpos($currentPath, $path) === 0) {
            return 'active';
        }
        
        return '';
    }
}

if (!function_exists('admin_can_access')) {
    /**
     * Check if admin can access specific feature
     * 
     * @param string $feature Feature name
     * @return bool
     */
    function admin_can_access(string $feature): bool
    {
        if (!is_admin_logged_in()) {
            return false;
        }
        
        $adminRole = get_admin_data('role');
        
        // Super admin can access everything
        if ($adminRole === 'super_admin') {
            return true;
        }
        
        // Define permissions for each role
        $permissions = [
            'admin' => [
                'activities_view',
                'activities_create', 
                'activities_edit',
                'activities_delete',
                'biodata_view',
                'biodata_create',
                'biodata_edit', 
                'biodata_delete',
                'education_view',
                'education_create',
                'education_edit',
                'education_delete',
                'profile_edit'
            ],
            'moderator' => [
                'activities_view',
                'activities_create',
                'activities_edit',
                'biodata_view',
                'biodata_edit',
                'education_view',
                'education_edit',
                'profile_edit'
            ]
        ];
        
        return in_array($feature, $permissions[$adminRole] ?? []);
    }
}