<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ProfileModel;

class ProfileManagement extends BaseController
{
    protected $profileModel;
    protected $session;

    public function __construct()
    {
        $this->profileModel = new ProfileModel();
        $this->session = \Config\Services::session();
        helper(['form', 'url']);
    }

    /**
     * Display profile management
     */
    public function index()
    {
        $data = [
            'title' => 'Kelola Profile',
            'profile' => $this->profileModel->getProfileWithSocialMedia()
        ];

        return view('admin/profile/index', $data);
    }

    /**
     * Edit profile form
     */
    public function edit($id = null)
    {
        $profile = $this->profileModel->getProfileWithSocialMedia($id);
        
        if (!$profile) {
            return redirect()->to('/admin/profile')->with('error', 'Profile tidak ditemukan.');
        }

        $data = [
            'title' => 'Edit Profile',
            'profile' => $profile,
            'validation' => \Config\Services::validation()
        ];

        return view('admin/profile/edit', $data);
    }

    /**
     * Update profile
     */
    public function update($id)
    {
        $profile = $this->profileModel->find($id);
        if (!$profile) {
            return redirect()->to('/admin/profile')->with('error', 'Profile tidak ditemukan.');
        }

        $rules = [
            'name' => 'required|min_length[3]|max_length[255]',
            'title' => 'max_length[255]',
            'description' => 'max_length[1000]',
            'email' => 'valid_email|max_length[255]',
            'phone' => 'max_length[50]',
            'address' => 'max_length[500]'
        ];

        // Only validate photo if it's uploaded
        $photo = $this->request->getFile('photo');
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            $rules['photo'] = 'uploaded[photo]|is_image[photo]|max_size[photo,2048]|ext_in[photo,jpg,jpeg,png,gif]';
        }

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $data = [
            'name' => $this->request->getPost('name'),
            'title' => $this->request->getPost('title'),
            'description' => $this->request->getPost('description'),
            'email' => $this->request->getPost('email'),
            'phone' => $this->request->getPost('phone'),
            'address' => $this->request->getPost('address'),
            'social_media' => [
                'instagram' => $this->request->getPost('instagram'),
                'linkedin' => $this->request->getPost('linkedin'),
                'github' => $this->request->getPost('github'),
                'twitter' => $this->request->getPost('twitter')
            ]
        ];

        // Handle photo upload
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            $newName = $photo->getRandomName();
            
            // Create uploads directory if not exists
            $uploadPath = FCPATH . 'uploads/profile/';
            if (!is_dir($uploadPath)) {
                if (!mkdir($uploadPath, 0755, true)) {
                    log_message('error', 'Failed to create upload directory: ' . $uploadPath);
                    return redirect()->back()->withInput()->with('error', 'Gagal membuat direktori upload.');
                }
            }
            
            // Check if directory is writable
            if (!is_writable($uploadPath)) {
                log_message('error', 'Upload directory not writable: ' . $uploadPath);
                return redirect()->back()->withInput()->with('error', 'Direktori upload tidak dapat ditulis.');
            }
            
            try {
                log_message('info', 'Attempting to move file to: ' . $uploadPath . $newName);
                
                if ($photo->move($uploadPath, $newName)) {
                    log_message('info', 'File uploaded successfully: ' . $newName);
                    
                    // Delete old photo if exists
                    if (!empty($profile['photo']) && file_exists($uploadPath . $profile['photo'])) {
                        unlink($uploadPath . $profile['photo']);
                        log_message('info', 'Old photo deleted: ' . $profile['photo']);
                    }
                    
                    $data['photo'] = $newName;
                } else {
                    $error = $photo->getErrorString();
                    log_message('error', 'Failed to move uploaded file: ' . $error);
                    return redirect()->back()->withInput()->with('error', 'Gagal mengupload foto: ' . $error);
                }
            } catch (\Exception $e) {
                log_message('error', 'Exception during file upload: ' . $e->getMessage());
                return redirect()->back()->withInput()->with('error', 'Error upload foto: ' . $e->getMessage());
            }
        } else if ($photo && !$photo->isValid()) {
            $error = $photo->getErrorString();
            log_message('error', 'Invalid file upload: ' . $error);
            return redirect()->back()->withInput()->with('error', 'File tidak valid: ' . $error);
        }

        try {
            if ($this->profileModel->updateProfile($id, $data)) {
                return redirect()->to('/admin/profile')->with('success', 'Profile berhasil diupdate.');
            } else {
                return redirect()->back()->withInput()->with('error', 'Gagal mengupdate profile.');
            }
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Delete photo
     */
    public function deletePhoto($id)
    {
        $profile = $this->profileModel->find($id);
        if (!$profile) {
            return redirect()->to('/admin/profile')->with('error', 'Profile tidak ditemukan.');
        }

        if (!empty($profile['photo'])) {
            $photoPath = FCPATH . 'uploads/profile/' . $profile['photo'];
            if (file_exists($photoPath)) {
                unlink($photoPath);
            }
            
            $this->profileModel->update($id, ['photo' => null]);
            return redirect()->to('/admin/profile')->with('success', 'Foto berhasil dihapus.');
        }

        return redirect()->to('/admin/profile')->with('error', 'Tidak ada foto untuk dihapus.');
    }
}