<?php 

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\EducationHistoryModel;
use CodeIgniter\Exceptions\PageNotFoundException;

class EducationAdmin extends BaseController
{
    protected $educationModel;
    protected $session;

    public function __construct()
    {
        $this->educationModel = new EducationHistoryModel();
        $this->session = \Config\Services::session();
    }

    public function index()
    {
        // Get search parameter
        $search = $this->request->getGet('search');
        
        // Get filter parameters
        $filter_jenjang = $this->request->getGet('filter_jenjang');
        
        // Get sorting parameters
        $sort_by = $this->request->getGet('sort_by') ?? 'tahun_mulai';
        $sort_order = $this->request->getGet('sort_order') ?? 'DESC';
        
        // Apply search
        if (!empty($search)) {
            $this->educationModel->groupStart()
                ->like('nama_sekolah', $search)
                ->orLike('jurusan', $search)
                ->groupEnd();
        }
        
        // Apply filter
        if (!empty($filter_jenjang)) {
            $this->educationModel->where('jenjang', $filter_jenjang);
        }
        
        // Apply sorting
        $this->educationModel->orderBy($sort_by, $sort_order);
        
        $data = [
            'title' => 'Kelola Pendidikan',
            'educations' => $this->educationModel->paginate(10),
            'pager' => $this->educationModel->pager,
            'search' => $search,
            'filter_jenjang' => $filter_jenjang,
            'sort_by' => $sort_by,
            'sort_order' => $sort_order,
        ];
        
        return view('admin/education/index', $data);
    }

    public function preview($id)
    {
        $education = $this->educationModel->where('id', $id)->first();
        
        if (!$education) {
            throw PageNotFoundException::forPageNotFound();
        }

        $data = [
            'title' => 'Preview Pendidikan',
            'education' => $education
        ];

        return view('admin/education/preview', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Pendidikan',
            'validation' => \Config\Services::validation()
        ];

        // lakukan validasi
        $validation = \Config\Services::validation();
        $validation->setRules([
            'jenjang' => 'required',
            'nama_sekolah' => 'required|min_length[3]|max_length[255]',
            'tahun_mulai' => 'required|integer',
            'tahun_selesai' => 'required|integer'
        ]);
        
        $isDataValid = $validation->withRequest($this->request)->run();

        // jika data valid, simpan ke database
        if ($isDataValid) {
            $this->educationModel->insert([
                "jenjang" => $this->request->getPost('jenjang'),
                "nama_sekolah" => $this->request->getPost('nama_sekolah'),
                "jurusan" => $this->request->getPost('jurusan'),
                "tahun_mulai" => $this->request->getPost('tahun_mulai'),
                "tahun_selesai" => $this->request->getPost('tahun_selesai'),
                "ipk" => $this->request->getPost('ipk'),
                "keterangan" => $this->request->getPost('keterangan')
            ]);
            
            return redirect()->to('/admin/education')->with('success', 'Data pendidikan berhasil ditambahkan.');
        }
        
        return view('admin/education/create', $data);
    }

    public function edit($id)
    {
        // ambil data yang akan diedit
        $education = $this->educationModel->where('id', $id)->first();
        
        if (!$education) {
            return redirect()->to('/admin/education')->with('error', 'Data pendidikan tidak ditemukan.');
        }

        $data = [
            'title' => 'Edit Pendidikan',
            'education' => $education,
            'validation' => \Config\Services::validation()
        ];
        
        // Check if this is a POST request (form submission)
        if ($this->request->getMethod() === 'POST') {
            // lakukan validasi data
            $validation = \Config\Services::validation();
            $validation->setRules([
                'id' => 'required',
                'jenjang' => 'required',
                'nama_sekolah' => 'required|min_length[3]|max_length[255]',
                'tahun_mulai' => 'required|integer',
                'tahun_selesai' => 'required|integer'
            ]);
            
            $isDataValid = $validation->withRequest($this->request)->run();
            
            // jika data valid, maka simpan ke database
            if ($isDataValid) {
                $updateData = [
                    "jenjang" => $this->request->getPost('jenjang'),
                    "nama_sekolah" => $this->request->getPost('nama_sekolah'),
                    "jurusan" => $this->request->getPost('jurusan'),
                    "tahun_mulai" => $this->request->getPost('tahun_mulai'),
                    "tahun_selesai" => $this->request->getPost('tahun_selesai'),
                    "ipk" => $this->request->getPost('ipk'),
                    "keterangan" => $this->request->getPost('keterangan')
                ];
                
                $this->educationModel->update($id, $updateData);
                return redirect()->to('/admin/education')->with('success', 'Data pendidikan berhasil diupdate.');
            }
            
            // If validation failed, set validation errors in data
            $data['validation'] = $validation;
        }

        return view('admin/education/edit', $data);
    }

    public function delete($id)
    {
        $education = $this->educationModel->where('id', $id)->first();
        
        if (!$education) {
            return redirect()->to('/admin/education')->with('error', 'Data pendidikan tidak ditemukan.');
        }
        
        $this->educationModel->delete($id);
        return redirect()->to('/admin/education')->with('success', 'Data pendidikan berhasil dihapus.');
    }
}