<?php

namespace App\Controllers;

use App\Models\EducationModel;
use CodeIgniter\Exceptions\PageNotFoundException;

class EducationAdmin extends BaseController
{
    private EducationModel $education;

    public function __construct()
    {
        $this->education = new EducationModel();
    }

    public function index(): string
    {
        $search    = trim((string) $this->request->getGet('q'));
        $degree    = trim((string) $this->request->getGet('degree'));
        $sort      = (string) $this->request->getGet('sort');
        $startYear = (int) $this->request->getGet('start_year');
        $endYear   = (int) $this->request->getGet('end_year');

        $query = $this->education;

        if ($search !== '') {
            $query->groupStart()
                ->like('institution', $search)
                ->orLike('field_of_study', $search)
                ->orLike('description', $search)
                ->groupEnd();
        }

        if ($degree !== '' && $degree !== 'semua') {
            $query->where('degree', $degree);
        }

        if ($startYear > 0) {
            $query->where('start_year >=', $startYear);
        }

        if ($endYear > 0) {
            $query->groupStart()
                ->where('end_year <=', $endYear)
                ->orWhere('end_year', null)
                ->groupEnd();
        }

        $sort = $sort !== '' ? $sort : 'start_desc';
        switch ($sort) {
            case 'start_asc':
                $query->orderBy('start_year', 'asc');
                break;
            case 'degree':
                $query->orderBy('degree', 'asc')->orderBy('start_year', 'desc');
                break;
            default:
                $query->orderBy('start_year', 'desc');
                $sort = 'start_desc';
        }

        $records = $query->paginate(10, 'education');

        $degreeOptions = ['SD', 'SMP', 'SMA/SMK', 'Diploma', 'Sarjana', 'Magister', 'Doktor', 'Lainnya'];

        return view('admin_list_education', [
            'records'       => $records,
            'pager'         => $this->education->pager,
            'degreeOptions' => $degreeOptions,
            'filters'       => [
                'search'     => $search,
                'degree'     => $degree !== '' ? $degree : 'semua',
                'start_year' => $startYear > 0 ? $startYear : '',
                'end_year'   => $endYear > 0 ? $endYear : '',
                'sort'       => $sort,
            ],
        ]);
    }

    public function create()
    {
        if ($this->request->is('post')) {
            $rules = $this->rules();

            if (! $this->validate($rules)) {
                return view('admin_form_education', [
                    'mode'       => 'create',
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $payload = $this->payloadFromRequest();

            $this->education->insert($payload);

            return redirect()
                ->to('/admin/pendidikan')
                ->with('message', 'Riwayat pendidikan berhasil ditambahkan.');
        }

        return view('admin_form_education', [
            'mode' => 'create',
        ]);
    }

    public function edit(int $id)
    {
        $record = $this->findOrFail($id);

        if ($this->request->is('post')) {
            $rules = $this->rules();

            if (! $this->validate($rules)) {
                return view('admin_form_education', [
                    'mode'       => 'edit',
                    'record'     => $record,
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $payload = $this->payloadFromRequest();

            $this->education->update($id, $payload);

            return redirect()
                ->to('/admin/pendidikan')
                ->with('message', 'Riwayat pendidikan berhasil diperbarui.');
        }

        return view('admin_form_education', [
            'mode'   => 'edit',
            'record' => $record,
        ]);
    }

    public function delete(int $id)
    {
        $this->findOrFail($id);
        $this->education->delete($id);

        return redirect()
            ->to('/admin/pendidikan')
            ->with('message', 'Riwayat pendidikan berhasil dihapus.');
    }

    private function rules(): array
    {
        return [
            'institution' => [
                'rules'  => 'required|string',
                'errors' => [
                    'required' => 'Institusi wajib diisi.',
                ],
            ],
            'degree' => [
                'rules'  => 'required|in_list[SD,SMP,SMA/SMK,Diploma,Sarjana,Magister,Doktor,Lainnya]',
                'errors' => [
                    'in_list' => 'Pilih jenjang pendidikan yang tersedia.',
                ],
            ],
            'start_year' => [
                'rules'  => 'required|integer|greater_than_equal_to[1900]|less_than_equal_to[2100]',
                'errors' => [
                    'required' => 'Tahun mulai wajib diisi.',
                ],
            ],
            'end_year' => [
                'rules'  => 'permit_empty|integer|greater_than_equal_to[1900]|less_than_equal_to[2100]',
            ],
            'field_of_study' => [
                'rules' => 'permit_empty|string',
            ],
            'city' => [
                'rules' => 'permit_empty|string',
            ],
            'description' => [
                'rules' => 'permit_empty|string',
            ],
        ];
    }

    private function payloadFromRequest(): array
    {
        $startYear = (int) $this->request->getPost('start_year');
        $endYear   = $this->request->getPost('end_year');
        $endYear   = $endYear === '' ? null : (int) $endYear;

        return [
            'institution'   => (string) $this->request->getPost('institution'),
            'degree'        => (string) $this->request->getPost('degree'),
            'field_of_study'=> (string) $this->request->getPost('field_of_study') ?: null,
            'city'          => (string) $this->request->getPost('city') ?: null,
            'start_year'    => $startYear,
            'end_year'      => $endYear,
            'description'   => (string) $this->request->getPost('description') ?: null,
        ];
    }

    private function findOrFail(int $id): array
    {
        $record = $this->education->find($id);
        if (! $record) {
            throw PageNotFoundException::forPageNotFound();
        }

        return $record;
    }
}
