# Presentasi Sistem Portal Berita - Admin Pegawai & Halaman About

## 📋 Daftar Isi
1. [Sistem Admin Pegawai](#sistem-admin-pegawai)
2. [Halaman About](#halaman-about)
3. [Teknologi yang Digunakan](#teknologi-yang-digunakan)
4. [Arsitektur Sistem](#arsitektur-sistem)
5. [Alur Kerja](#alur-kerja)

---

## 🏢 Sistem Admin Pegawai

### Pengenalan
Sistem Admin Pegawai adalah modul yang memungkinkan administrator untuk mengelola data pegawai perusahaan secara lengkap. Sistem ini menyediakan fitur CRUD (Create, Read, Update, Delete) untuk data pegawai beserta foto profil.

### Fitur Utama

#### 1. **Daftar Pegawai (List)**
- Menampilkan semua data pegawai dalam bentuk tabel
- Menampilkan foto profil, nama, tanggal lahir, divisi, dan jenis kelamin
- Fitur pencarian dan filter (dapat dikembangkan)
- Tombol aksi untuk Edit dan Hapus

#### 2. **Tambah Pegawai Baru (Create)**
- Form input untuk menambahkan pegawai baru
- Validasi data sebelum disimpan
- Upload foto profil (opsional)
- Validasi format dan ukuran file

#### 3. **Edit Pegawai (Update)**
- Form untuk mengubah data pegawai yang sudah ada
- Dapat mengganti foto profil
- Dapat menghapus foto profil yang sudah ada
- Validasi data yang sama seperti form tambah

#### 4. **Hapus Pegawai (Delete)**
- Menghapus data pegawai dari database
- Otomatis menghapus foto profil yang terkait
- Konfirmasi sebelum menghapus untuk mencegah kesalahan

### Struktur Database

#### Tabel: `pegawai`

```sql
- id_pegawai (INT, Primary Key, Auto Increment)
  → ID unik untuk setiap pegawai
  
- nama_pegawai (VARCHAR 150)
  → Nama lengkap pegawai
  
- tanggal_lahir (DATE)
  → Tanggal lahir pegawai
  
- jenis_kelamin (ENUM: 'laki-laki', 'perempuan')
  → Jenis kelamin pegawai
  
- divisi (ENUM: 'Direktur Utama', 'IT', 'Produksi', 'Marketing', 'HR')
  → Divisi tempat pegawai bekerja
  
- foto_pegawai (VARCHAR 255, nullable)
  → Path/lokasi file foto pegawai di server
  
- created_at (DATETIME)
  → Waktu data pertama kali dibuat
  
- updated_at (DATETIME)
  → Waktu data terakhir diupdate
```

**Penjelasan:**
- `id_pegawai` adalah kunci utama yang otomatis bertambah setiap ada data baru
- `foto_pegawai` menyimpan path relatif ke file foto (contoh: `uploads/pegawai/abc123.jpg`)
- `created_at` dan `updated_at` otomatis diisi oleh sistem untuk tracking

### Komponen Sistem

#### 1. **Controller: `PerusahaanAdmin.php`**

Controller ini menangani semua request yang berkaitan dengan manajemen pegawai.

**Fungsi-fungsi utama:**

```php
// Menampilkan daftar semua pegawai
public function index(): string
```
**Penjelasan:** Fungsi ini mengambil semua data pegawai dari database, mengurutkannya berdasarkan ID, lalu menampilkannya di halaman daftar pegawai.

```php
// Menampilkan form tambah pegawai atau menyimpan data baru
public function create()
```
**Penjelasan:** 
- Jika user mengirim form (POST), sistem akan:
  1. Validasi data yang diinput
  2. Upload foto jika ada
  3. Simpan data ke database
  4. Redirect ke halaman daftar dengan pesan sukses
- Jika user hanya membuka halaman (GET), sistem menampilkan form kosong

```php
// Menampilkan form edit atau menyimpan perubahan
public function edit(int $id)
```
**Penjelasan:**
- Mencari data pegawai berdasarkan ID
- Jika data tidak ditemukan, tampilkan error 404
- Jika POST: validasi, update data, dan simpan
- Jika GET: tampilkan form dengan data yang sudah ada

```php
// Menghapus data pegawai
public function delete(int $id)
```
**Penjelasan:**
- Mencari data pegawai
- Hapus foto dari server jika ada
- Hapus data dari database
- Redirect dengan pesan sukses

**Fungsi Helper (Private):**

```php
// Menyimpan foto yang diupload
private function storePegawaiPhoto(?UploadedFile $photoFile): ?string
```
**Penjelasan:**
- Menerima file yang diupload
- Validasi apakah file valid (tipe gambar, ukuran maksimal 2MB)
- Simpan file ke folder `public/uploads/pegawai/`
- Generate nama file random untuk keamanan
- Return path relatif file (contoh: `uploads/pegawai/xyz789.jpg`)

```php
// Menghapus foto dari server
private function deletePegawaiPhoto(?string $relativePath): void
```
**Penjelasan:**
- Menerima path file yang akan dihapus
- Hapus file dari server jika file tersebut ada
- Digunakan saat edit (ganti foto) atau delete pegawai

```php
// Aturan validasi data
private function pegawaiRules(): array
```
**Penjelasan:**
- Mendefinisikan aturan validasi untuk setiap field
- Menentukan pesan error yang akan ditampilkan jika validasi gagal
- Memastikan data yang masuk sesuai format yang diinginkan

#### 2. **Model: `PegawaiModel.php`**

Model ini menghubungkan aplikasi dengan tabel `pegawai` di database.

```php
protected $table = 'pegawai';
```
**Penjelasan:** Menentukan nama tabel yang digunakan

```php
protected $primaryKey = 'id_pegawai';
```
**Penjelasan:** Menentukan kolom yang menjadi primary key

```php
protected $allowedFields = [
    'nama_pegawai',
    'tanggal_lahir',
    'jenis_kelamin',
    'divisi',
    'foto_pegawai',
];
```
**Penjelasan:** Daftar field yang diizinkan untuk diisi/diubah. Field lain seperti `id_pegawai`, `created_at`, `updated_at` dikelola otomatis oleh sistem.

```php
protected $useTimestamps = true;
```
**Penjelasan:** Aktifkan fitur auto-timestamp, sehingga `created_at` dan `updated_at` otomatis diisi saat create/update.

### Validasi Data

Sistem menggunakan validasi CodeIgniter untuk memastikan data yang masuk valid:

1. **Nama Pegawai**
   - Wajib diisi
   - Harus berupa teks (string)

2. **Tanggal Lahir**
   - Wajib diisi
   - Harus format tanggal yang valid (YYYY-MM-DD)

3. **Jenis Kelamin**
   - Wajib dipilih
   - Hanya boleh: "laki-laki" atau "perempuan"

4. **Divisi**
   - Wajib dipilih
   - Hanya boleh: "Direktur Utama", "IT", "Produksi", "Marketing", atau "HR"

5. **Foto Pegawai**
   - Opsional (boleh kosong)
   - Jika diisi, harus berupa gambar
   - Format: JPG, JPEG, PNG, atau WEBP
   - Ukuran maksimal: 2 MB

### Keamanan

1. **CSRF Protection**
   - Semua form dilindungi dengan CSRF token
   - Mencegah serangan Cross-Site Request Forgery

2. **Input Validation**
   - Semua input divalidasi sebelum disimpan
   - Mencegah SQL Injection dan XSS

3. **File Upload Security**
   - Validasi tipe file (hanya gambar)
   - Validasi ukuran file (maksimal 2MB)
   - Nama file di-random untuk mencegah path traversal

4. **Authentication**
   - Hanya admin yang sudah login yang bisa mengakses
   - Dilindungi oleh `AuthFilter`

---

## 📄 Halaman About

### Pengenalan
Halaman About adalah halaman publik yang menampilkan informasi tentang perusahaan, termasuk visi, misi, nilai-nilai, timeline perusahaan, dan struktur tim berdasarkan divisi.

### Fitur Utama

#### 1. **Hero Section**
- Banner perusahaan (gambar besar)
- Logo perusahaan
- Judul dan deskripsi perusahaan
- Tagline dan nilai-nilai inti

#### 2. **Statistik Perusahaan**
- Menampilkan metrik penting (jumlah proyek, talenta, kepuasan klien)
- Tampilan dalam bentuk card yang menarik

#### 3. **Misi & Nilai Inti**
- Penjelasan misi perusahaan
- Daftar nilai-nilai inti yang dianut

#### 4. **Timeline Perusahaan**
- Perjalanan perusahaan dari tahun ke tahun
- Pencapaian-pencapaian penting

#### 5. **Struktur Tim (Team Structure)**
- Menampilkan pegawai berdasarkan divisi
- Setiap pegawai ditampilkan dalam card dengan:
  - Foto profil (atau initial jika tidak ada foto)
  - Nama pegawai
  - Jenis kelamin
  - Tanggal lahir

### Komponen Sistem

#### 1. **Controller: `Page.php`**

```php
public function about()
```
**Penjelasan:**
- Mengambil semua data pegawai dari database
- Mengurutkan berdasarkan ID
- Mengirim data ke view `about.php`
- Menonaktifkan header default (karena halaman about punya header sendiri)

#### 2. **View: `about.php`**

View ini menampilkan halaman About dengan berbagai section.

**Fungsi-fungsi Helper di View:**

```php
$resolveAsset = static function ($candidates) {
    // Mencari file asset (banner, logo) dari beberapa lokasi
    // Jika ditemukan, return path dengan cache-busting (versi)
    // Jika tidak ditemukan, return path default
}
```
**Penjelasan:** 
- Mencari file gambar (banner, logo) dari beberapa kemungkinan lokasi
- Jika file ditemukan, menambahkan timestamp untuk cache-busting (agar browser selalu load versi terbaru)
- Jika tidak ditemukan, tetap return path untuk fallback

```php
$makeInitials = static function (string $text, int $length = 2): string {
    // Mengambil huruf pertama dari setiap kata
    // Contoh: "John Doe" → "JD"
    // Contoh: "MADANI YATSI CREATIVE" → "MY"
}
```
**Penjelasan:**
- Jika pegawai tidak punya foto, sistem akan menampilkan initial (huruf pertama)
- Mengambil huruf pertama dari setiap kata dalam nama
- Maksimal 2 huruf (default)

**Struktur Data Pegawai di Halaman:**

```php
// Mengelompokkan pegawai berdasarkan divisi
$groupedPegawai = [];
foreach ($pegawai as $item) {
    $divisi = $item['divisi'] ?? 'Lainnya';
    $groupedPegawai[$divisi][] = $item;
}
```
**Penjelasan:**
- Data pegawai dikelompokkan berdasarkan divisi
- Setiap divisi ditampilkan dalam section terpisah
- Memudahkan user melihat struktur organisasi

**Display Logic:**

```php
// Jika ada foto, tampilkan foto
<?php if ($foto['exists']): ?>
    <img src="<?= esc($foto['url']) ?>" alt="Foto pegawai">
<?php else: ?>
    // Jika tidak ada foto, tampilkan initial
    <div class="team-card__placeholder"><?= esc($initial) ?></div>
<?php endif; ?>
```
**Penjelasan:**
- Sistem mengecek apakah pegawai punya foto
- Jika ada: tampilkan foto
- Jika tidak: tampilkan initial dalam kotak berwarna

### Responsive Design

Halaman About dirancang responsif untuk berbagai ukuran layar:

1. **Desktop (>992px)**
   - Layout 2 kolom untuk hero section
   - Grid 4 kolom untuk kartu pegawai

2. **Tablet (768px - 992px)**
   - Layout 1 kolom untuk hero section
   - Grid 2-3 kolom untuk kartu pegawai

3. **Mobile (<768px)**
   - Layout 1 kolom penuh
   - Grid 1 kolom untuk kartu pegawai
   - Font size dan spacing disesuaikan

### CSS Styling

Halaman About menggunakan CSS custom dengan:

1. **Gradient Background**
   - Background dengan gradient warna untuk efek modern
   - Overlay untuk readability

2. **Card Design**
   - Kartu dengan shadow dan border radius
   - Hover effect untuk interaktivitas

3. **Typography**
   - Font size responsif menggunakan `clamp()`
   - Line height yang nyaman untuk dibaca

4. **Spacing**
   - Margin dan padding yang konsisten
   - Gap antar elemen yang proporsional

---

## 🛠 Teknologi yang Digunakan

### Backend

1. **PHP 8.1+**
   - Bahasa pemrograman utama
   - Type hints untuk keamanan tipe data

2. **CodeIgniter 4**
   - Framework PHP untuk pengembangan web
   - MVC (Model-View-Controller) pattern
   - Built-in security features

3. **MySQL/MariaDB**
   - Database untuk menyimpan data
   - Relational database management system

### Frontend

1. **HTML5**
   - Struktur halaman web
   - Semantic HTML untuk SEO

2. **CSS3**
   - Styling dan layout
   - Flexbox dan Grid untuk responsive design
   - Custom properties (CSS variables)

3. **JavaScript (Vanilla)**
   - Interaktivitas di client-side
   - Form validation
   - Sidebar toggle untuk mobile

4. **Bootstrap 5**
   - CSS framework untuk komponen UI
   - Grid system untuk layout
   - Utility classes

### Security

1. **CSRF Protection**
   - Token untuk mencegah Cross-Site Request Forgery
   - Otomatis di-generate oleh CodeIgniter

2. **XSS Protection**
   - Fungsi `esc()` untuk escape output
   - Mencegah script injection

3. **SQL Injection Protection**
   - Query Builder CodeIgniter
   - Parameterized queries

4. **File Upload Security**
   - Validasi tipe file
   - Validasi ukuran file
   - Random filename

### Tools & Libraries

1. **Composer**
   - Dependency management untuk PHP

2. **Git**
   - Version control system

---

## 🏗 Arsitektur Sistem

### MVC Pattern

Sistem menggunakan pola MVC (Model-View-Controller):

```
Request → Controller → Model → Database
                ↓
              View → Response
```

**Penjelasan:**
1. User mengirim request (contoh: klik tombol "Tambah Pegawai")
2. Request diterima oleh Controller (`PerusahaanAdmin`)
3. Controller meminta data ke Model (`PegawaiModel`)
4. Model berinteraksi dengan Database
5. Controller mengirim data ke View
6. View menampilkan halaman HTML ke user

### File Structure

```
app/
├── Controllers/
│   ├── PerusahaanAdmin.php    → Controller untuk admin pegawai
│   └── Page.php                → Controller untuk halaman About
├── Models/
│   └── PegawaiModel.php        → Model untuk tabel pegawai
├── Views/
│   ├── admin_list_pegawai.php  → Halaman daftar pegawai
│   ├── admin_create_pegawai.php → Form tambah pegawai
│   ├── admin_edit_pegawai.php  → Form edit pegawai
│   └── about.php               → Halaman About
└── Database/
    └── Migrations/
        └── CreatePegawai.php   → Skema database tabel pegawai
```

### Alur Data

#### Saat Menambah Pegawai Baru:

```
1. User mengisi form → Submit
2. Browser mengirim POST request ke /admin/perusahaan/new
3. Controller PerusahaanAdmin::create() menerima request
4. Controller memvalidasi data
5. Jika valid:
   - Upload foto (jika ada) → Simpan ke public/uploads/pegawai/
   - Simpan data ke database melalui Model
   - Redirect ke halaman daftar dengan pesan sukses
6. Jika tidak valid:
   - Tampilkan form kembali dengan pesan error
```

#### Saat Menampilkan Halaman About:

```
1. User mengakses /about
2. Controller Page::about() dipanggil
3. Controller mengambil data pegawai dari Model
4. Controller mengirim data ke View about.php
5. View menampilkan:
   - Hero section dengan banner/logo
   - Statistik perusahaan
   - Misi & nilai
   - Timeline
   - Struktur tim (dikelompokkan per divisi)
6. Browser menampilkan halaman HTML
```

---

## 🔄 Alur Kerja

### Admin Pegawai - Tambah Data

```
1. Admin login → Masuk ke dashboard
2. Klik menu "Perusahaan" → Lihat daftar pegawai
3. Klik "Tambah Pegawai" → Form tambah muncul
4. Isi form:
   - Nama pegawai (wajib)
   - Tanggal lahir (wajib)
   - Jenis kelamin (wajib)
   - Divisi (wajib)
   - Foto pegawai (opsional)
5. Klik "Simpan Data"
6. Sistem validasi:
   - Jika ada error → Tampilkan pesan error
   - Jika valid → Simpan ke database
7. Redirect ke halaman daftar dengan pesan sukses
```

### Admin Pegawai - Edit Data

```
1. Dari halaman daftar → Klik "Edit" pada pegawai tertentu
2. Form edit muncul dengan data yang sudah ada
3. Ubah data yang ingin diubah
4. Untuk foto:
   - Bisa upload foto baru (akan mengganti yang lama)
   - Bisa hapus foto (centang "Hapus foto")
5. Klik "Simpan Perubahan"
6. Sistem update data di database
7. Redirect dengan pesan sukses
```

### Admin Pegawai - Hapus Data

```
1. Dari halaman daftar → Klik "Hapus" pada pegawai tertentu
2. Konfirmasi muncul: "Hapus data pegawai ini secara permanen?"
3. Jika "OK":
   - Sistem hapus foto dari server (jika ada)
   - Sistem hapus data dari database
   - Redirect dengan pesan sukses
4. Jika "Cancel" → Tidak ada yang terjadi
```

### Halaman About - Menampilkan Data

```
1. User mengakses /about
2. Sistem mengambil semua data pegawai dari database
3. Data dikelompokkan berdasarkan divisi
4. Untuk setiap pegawai:
   - Cek apakah ada foto
   - Jika ada: tampilkan foto
   - Jika tidak: tampilkan initial (huruf pertama nama)
5. Tampilkan dalam card yang rapi dan responsif
```

---

## 📊 Diagram Alur

### Flowchart - Tambah Pegawai

```
START
  ↓
Admin Login?
  ↓ YES
Akses Menu Perusahaan
  ↓
Klik "Tambah Pegawai"
  ↓
Form Tambah Muncul
  ↓
Isi Data Form
  ↓
Klik "Simpan"
  ↓
Validasi Data
  ↓
Valid? ────NO──→ Tampilkan Error
  ↓ YES
Upload Foto? ──YES→ Upload ke Server
  ↓ NO
Simpan ke Database
  ↓
Tampilkan Pesan Sukses
  ↓
END
```

### Flowchart - Tampilkan About

```
START
  ↓
User Akses /about
  ↓
Controller Ambil Data Pegawai
  ↓
Kelompokkan Berdasarkan Divisi
  ↓
Loop Setiap Pegawai:
  ├─ Ada Foto? ─YES→ Tampilkan Foto
  └─ Tidak Ada? ─NO→ Tampilkan Initial
  ↓
Render HTML dengan Data
  ↓
Kirim ke Browser
  ↓
END
```

---

## 🎯 Kesimpulan

### Sistem Admin Pegawai
- Sistem yang lengkap untuk mengelola data pegawai
- Fitur CRUD lengkap dengan validasi
- Upload dan manajemen foto profil
- Keamanan yang baik dengan CSRF dan validasi input

### Halaman About
- Halaman publik yang informatif
- Menampilkan struktur tim berdasarkan divisi
- Design modern dan responsif
- Integrasi dengan data pegawai dari database

### Teknologi
- Menggunakan CodeIgniter 4 sebagai framework
- PHP 8.1+ untuk backend
- MySQL untuk database
- HTML, CSS, JavaScript untuk frontend
- Bootstrap 5 untuk UI components

### Keamanan
- CSRF protection
- XSS protection
- SQL injection protection
- File upload validation
- Authentication required untuk admin

---

**Dibuat untuk presentasi sistem Portal Berita CodeIgniter 4**




