<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\EducationModel;

class EducationController extends BaseController
{
    protected $educationModel;

    public function __construct()
    {
        $this->educationModel = new EducationModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search');
        $start_year = $this->request->getGet('start_year');
        $sort = $this->request->getGet('sort') ?? 'newest';
        $order = $this->request->getGet('order') ?? 'DESC';

        $education = $this->educationModel;

        if ($search) {
            $education = $education->like('institution', $search)->orLike('description', $search);
        }

        if ($start_year) {
            $education = $education->where('start_year', $start_year);
        }

        // Sorting
        $allowedColumns = ['institution', 'start_year', 'end_year'];
        if (in_array($sort, $allowedColumns)) {
            $education = $education->orderBy($sort, $order);
        } elseif ($sort === 'oldest') {
            $education = $education->orderBy('start_year', 'ASC');
        } else {
            // Default newest
            $education = $education->orderBy('start_year', 'DESC');
        }

        $data = [
            'education' => $education->paginate(10),
            'pager' => $this->educationModel->pager,
            'search' => $search,
            'start_year' => $start_year,
            'sort' => $sort,
            'order' => $order,
        ];

        return view('admin/education/index', $data);
    }

    public function new()
    {
        return view('admin/education/form', ['title' => 'New Education']);
    }

    public function create()
    {
        if (
            !$this->validate([
                'institution' => 'required',
                'start_year' => 'required|numeric',
            ])
        ) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $this->educationModel->save([
            'institution' => $this->request->getPost('institution'),
            'start_year' => $this->request->getPost('start_year'),
            'end_year' => $this->request->getPost('end_year') ?: null,
            'description' => $this->request->getPost('description'),
        ]);

        return redirect()->to('/admin/education')->with('message', 'Education record created successfully');
    }

    public function edit($id)
    {
        $item = $this->educationModel->find($id);
        if (!$item) {
            return redirect()->to('/admin/education')->with('error', 'Record not found');
        }

        return view('admin/education/form', ['item' => $item, 'title' => 'Edit Education']);
    }

    public function update($id)
    {
        if (
            !$this->validate([
                'institution' => 'required',
                'start_year' => 'required|numeric',
            ])
        ) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $this->educationModel->save([
            'id' => $id,
            'institution' => $this->request->getPost('institution'),
            'start_year' => $this->request->getPost('start_year'),
            'end_year' => $this->request->getPost('end_year') ?: null,
            'description' => $this->request->getPost('description'),
        ]);

        return redirect()->to('/admin/education')->with('message', 'Record updated successfully');
    }

    public function delete($id)
    {
        $this->educationModel->delete($id);
        return redirect()->to('/admin/education')->with('message', 'Record deleted successfully');
    }
}
